/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.utils;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.api.dsl.DslResolvingContext.getDefault;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.connection.HasConnectionProviderModels;
import org.mule.runtime.api.meta.model.construct.HasConstructModels;
import org.mule.runtime.api.meta.model.operation.HasOperationModels;
import org.mule.runtime.api.meta.model.source.HasSourceModels;
import org.mule.runtime.api.meta.model.util.ExtensionWalker;
import org.mule.runtime.api.util.Reference;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;

import com.google.common.collect.ImmutableSet;

import java.net.MalformedURLException;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;

import org.apache.commons.io.FileUtils;

/**
 * Utility class for Extension model related services
 */
public final class ExtensionModelUtils {

  private ExtensionModelUtils() {}

  public static DslResolvingContext createDslResolvingContext(List<ExtensionModel> muleModels,
                                                              List<ExtensionModel> artifactPlugins) {
    return getDefault(ImmutableSet.<ExtensionModel>builder()
        .addAll(muleModels)
        .addAll(artifactPlugins)
        .build());
  }

  public static List<ExtensionModel> resolveExtensionModels(MavenClient mavenClient,
                                                            MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                                            List<ArtifactDescriptor> pluginArtifactDescriptors) {
    List<BundleDependency> pluginDependencies = mavenClient
        .resolvePluginBundleDescriptorsDependencies(pluginArtifactDescriptors
            .stream().map(descriptor -> new BundleDescriptor.Builder()
                .setArtifactId(descriptor.getArtifactId())
                .setClassifier(descriptor.getClassifier())
                .setGroupId(descriptor.getGroupId())
                .setType(descriptor.getExtension())
                .setVersion(descriptor.getVersion()).build())
            .collect(toList()));

    return pluginDependencies.stream()
        .map(bundleDependency -> muleRuntimeExtensionModelProvider
            .getExtensionModel(toArtifactBundleDescriptor(bundleDependency.getDescriptor())).orElse(null))
        .filter(Objects::nonNull)
        .collect(toList());
  }

  public static NamedObject find(org.mule.runtime.api.meta.model.ExtensionModel extension, ConfigurationModel config) {
    return doFind(extension.getName(), config.getName(), "CONFIG",
                  reference -> new ExtensionWalker() {

                    @Override
                    protected void onConfiguration(org.mule.runtime.api.meta.model.config.ConfigurationModel model) {
                      if (model.getName().equals(config.getName())) {
                        reference.set(model);
                        stop();
                      }
                    }
                  }.walk(extension));
  }

  public static NamedObject find(org.mule.runtime.api.meta.model.ExtensionModel extension, ConnectionProviderModel component) {
    return doFind(extension.getName(), component.getName(), "CONNECTION",
                  reference -> new ExtensionWalker() {

                    @Override
                    protected void onConnectionProvider(HasConnectionProviderModels owner,
                                                        org.mule.runtime.api.meta.model.connection.ConnectionProviderModel model) {
                      if (model.getName().equals(component.getName())) {
                        reference.set(model);
                        stop();
                      }
                    }
                  }.walk(extension));
  }

  public static NamedObject find(org.mule.runtime.api.meta.model.ExtensionModel extension, ConstructModel construct) {
    return doFind(extension.getName(), construct.getName(), "CONSTRUCT",
                  reference -> new ExtensionWalker() {

                    @Override
                    protected void onConstruct(HasConstructModels owner,
                                               org.mule.runtime.api.meta.model.construct.ConstructModel model) {
                      if (model.getName().equals(construct.getName())) {
                        reference.set(model);
                        stop();
                      }
                    }
                  }.walk(extension));
  }

  public static NamedObject find(org.mule.runtime.api.meta.model.ExtensionModel extension, OperationModel operation) {
    return doFind(extension.getName(), operation.getName(), "OPERATION",
                  reference -> new ExtensionWalker() {

                    @Override
                    protected void onOperation(HasOperationModels owner,
                                               org.mule.runtime.api.meta.model.operation.OperationModel model) {
                      if (model.getName().equals(operation.getName())) {
                        reference.set(model);
                        stop();
                      }
                    }

                  }.walk(extension));
  }

  public static NamedObject find(org.mule.runtime.api.meta.model.ExtensionModel extension, SourceModel source) {
    return doFind(extension.getName(), source.getName(), "SOURCE",
                  reference -> new ExtensionWalker() {

                    @Override
                    protected void onSource(HasSourceModels owner, org.mule.runtime.api.meta.model.source.SourceModel model) {
                      if (model.getName().equals(source.getName())) {
                        reference.set(model);
                        stop();
                      }
                    }

                  }.walk(extension));
  }

  private static NamedObject doFind(String extension, String component, String kind, Consumer<Reference<NamedObject>> consumer) {
    Reference<NamedObject> reference = new Reference<>();
    consumer.accept(reference);
    NamedObject result = reference.get();
    if (result == null) {
      throw new IllegalStateException(format("Component [%s] of type [%s] was not found in extension [%s]",
                                             component, kind, extension));
    }
    return result;
  }

  private static org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor toArtifactBundleDescriptor(BundleDescriptor bundleDescriptor) {
    org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor.Builder builder =
        new org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor.Builder()
            .setGroupId(bundleDescriptor.getGroupId())
            .setArtifactId(bundleDescriptor.getArtifactId())
            .setVersion(bundleDescriptor.getVersion())
            .setType(bundleDescriptor.getType());
    bundleDescriptor.getClassifier().ifPresent(classifier -> builder.setClassifier(classifier));
    return builder.build();
  }

}
