/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptySet;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.codehaus.plexus.util.FileUtils.toFile;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.classloader.RegionClassLoader;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.List;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@Feature("DataSense")
@Story("ApplicationModelFactory should be able to create the ApplicationModel either from a remote location, from a local file system with jar or expanded content for application")
@RunWith(MockitoJUnitRunner.class)
public class DefaultApplicationModelFactoryTestCase {

  @Mock
  private RegionClassLoader regionClassLoader;

  @Test
  @Description("Checks that an application model that uses CE components is created from an expanded folder")
  public void createApplicationModel() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/deploy-json-descriptor");

    assertApplicationModel(new DefaultApplicationModelFactory()
        .createApplicationModel(applicationDescriptor.getFirst(),
                                emptySet(),
                                new URLClassLoader(new URL[] {applicationDescriptor.getSecond().toURI().toURL()},
                                                   regionClassLoader),
                                null));
  }

  @Test
  @Description("Checks that an application model that references properties can be resolved with the class loader")
  public void createApplicationModelWithPropertiesResolved() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/deploy-placeholders");
    assertApplicationModel(new DefaultApplicationModelFactory()
        .createApplicationModel(applicationDescriptor.getFirst(),
                                emptySet(),
                                new URLClassLoader(new URL[] {applicationDescriptor.getSecond().toURI().toURL()},
                                                   regionClassLoader),

                                null));
  }

  private Pair<ApplicationDescriptor, File> createApplicationDescriptor(String applicationRelativeLocation) throws IOException {
    File applicationRootFile = toFile(this.getClass().getClassLoader().getResource(applicationRelativeLocation));
    final MuleApplicationModel muleApplicationModel = new MuleApplicationModelJsonSerializer().deserialize(
                                                                                                           readFileToString(new File(new File(new File(applicationRootFile,
                                                                                                                                                       "META-INF"),
                                                                                                                                              "mule-artifact"),
                                                                                                                                     "mule-artifact.json")));
    ApplicationDescriptor applicationDescriptor = mock(ApplicationDescriptor.class);
    when(applicationDescriptor.getConfigResources()).thenReturn(muleApplicationModel.getConfigs());
    return new Pair(applicationDescriptor, applicationRootFile);
  }

  @Test
  @Description("Checks that an application model that has properties that cannot be resolved is created")
  public void createApplicationModelWithPropertiesNotResolved() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/deploy-placeholders");

    assertApplicationModel(new DefaultApplicationModelFactory()
        .createApplicationModel(applicationDescriptor.getFirst(),
                                emptySet(),
                                new URLClassLoader(new URL[] {applicationDescriptor.getSecond().toURI().toURL()},
                                                   regionClassLoader),
                                null));
  }

  @Test
  @Description("Checks that an application model that uses CE components is created from an expanded folder")
  public void createApplicationModelEE() throws Exception {
    Pair<ApplicationDescriptor, File> applicationDescriptor = createApplicationDescriptor("applications/datasense-static");

    assertApplicationModelEE(
                             new DefaultApplicationModelFactory()
                                 .createApplicationModel(applicationDescriptor.getFirst(), emptySet(),
                                                         new URLClassLoader(new URL[] {
                                                             applicationDescriptor.getSecond().toURI().toURL()},
                                                                            regionClassLoader),
                                                         null));
  }

  protected void assertApplicationModel(Optional<ApplicationModel> applicationModelOptional) {
    assertThat(applicationModelOptional.isPresent(), is(true));

    ApplicationModel applicationModel = applicationModelOptional.get();
    assertThat(new File(applicationModel.findTopLevelNamedComponent("simpleFlow").get().getConfigFileName().get()).getName(),
               equalTo("myApp.xml"));
    assertThat(new File(applicationModel.findTopLevelNamedComponent("simpleFlowExtended").get().getConfigFileName().get())
        .getName(),
               equalTo("myApp-extended.xml"));
  }

  protected void assertApplicationModelEE(Optional<ApplicationModel> applicationModelOptional) {
    assertThat(applicationModelOptional.isPresent(), is(true));

    ApplicationModel applicationModel = applicationModelOptional.get();
    List<ComponentModel> componentModels =
        applicationModel.findTopLevelNamedComponent("staticDataSenseFlow").get().getInnerComponents();
    assertThat(componentModels.get(0).getComponentLocation().getLocation(), is("staticDataSenseFlow/processors/0"));
    assertThat(componentModels.get(1).getComponentLocation().getLocation(), is("staticDataSenseFlow/processors/1"));

    componentModels = applicationModel.findTopLevelNamedComponent("staticDataSenseFlowEE").get().getInnerComponents();
    assertThat(componentModels.get(0).getComponentLocation().getLocation(), is("staticDataSenseFlowEE/processors/0"));
    assertThat(componentModels.get(1).getComponentLocation().getLocation(), is("staticDataSenseFlowEE/processors/1"));
  }

}
