/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptyList;
import static org.mule.tooling.client.internal.LocationFactory.toLocationDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataFailuresDTO;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataCacheProvider;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.Pair;
import org.mule.tooling.client.api.datasense.ImmutableMetadataCacheKeyInfo;
import org.mule.tooling.client.api.datasense.ImmutableMetadataResult;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataFailure;
import org.mule.tooling.client.internal.datasense.DataSenseArtifact;
import org.mule.tooling.client.internal.serialization.DefaultArrayTypeSerializer;
import org.mule.tooling.client.internal.serialization.KryoFactory;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

public class DataSenseMetadataCacheAdapter implements DataSenseMetadataCacheProvider {

  private DataSenseArtifact dataSenseArtifact;

  private Kryo kryo;

  public DataSenseMetadataCacheAdapter(DataSenseArtifact dataSenseArtifact) {
    this.dataSenseArtifact = dataSenseArtifact;

    this.kryo = KryoFactory.createKryo();
    kryo.addDefaultSerializer(DefaultArrayType.class, new DefaultArrayTypeSerializer());
  }

  public String serialize(Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        kryo.writeClassAndObject(output, object);
      }
      return Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  public <T> T deserialize(String content) {
    try (Input input = new Input(new ByteArrayInputStream(Base64.getDecoder().decode(content)))) {
      return (T) kryo.readClassAndObject(input);
    }
  }

  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(String componentId, Location location,
                                                                                          Long timestamp,
                                                                                          Callable<MetadataResult<ComponentMetadataDescriptor<OperationModel>>> callable) {
    Pair<MetadataCache, Map<String, String>> input = getDataSenseCache(location);

    String result =
        (String) input.getFirst().getOperationMetadata(new ImmutableMetadataCacheKeyInfo(componentId,
                                                                                         toLocationDTO(location).toString(),
                                                                                         timestamp, input.getSecond()),
                                                       () -> {
                                                         final MetadataResult<ComponentMetadataDescriptor<OperationModel>> runtimeMetadataResult =
                                                             callable.call();
                                                         List<MetadataFailure> failures = emptyList();
                                                         if (!runtimeMetadataResult.isSuccess()) {
                                                           failures = toMetadataFailuresDTO(runtimeMetadataResult.getFailures());
                                                         }
                                                         return new ImmutableMetadataResult(runtimeMetadataResult
                                                             .isSuccess(), serialize(runtimeMetadataResult), failures);
                                                       });

    return (MetadataResult<ComponentMetadataDescriptor<OperationModel>>) deserialize(result);
  }

  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(String componentId, Location location,
                                                                                    Long timestamp,
                                                                                    Callable<MetadataResult<ComponentMetadataDescriptor<SourceModel>>> callable) {
    Pair<MetadataCache, Map<String, String>> input = getDataSenseCache(location);

    String result =
        (String) input.getFirst()
            .getSourceMetadata(new ImmutableMetadataCacheKeyInfo(componentId, toLocationDTO(location).toString(), timestamp,
                                                                 input.getSecond()),
                               () -> {
                                 final MetadataResult<ComponentMetadataDescriptor<SourceModel>> runtimeMetadataResult =
                                     callable.call();
                                 List<MetadataFailure> failures = emptyList();
                                 if (!runtimeMetadataResult.isSuccess()) {
                                   failures = toMetadataFailuresDTO(runtimeMetadataResult.getFailures());
                                 }
                                 return new ImmutableMetadataResult(runtimeMetadataResult.isSuccess(),
                                                                    serialize(runtimeMetadataResult), failures);
                               });

    return (MetadataResult<ComponentMetadataDescriptor<SourceModel>>) deserialize(result);
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(String componentId, Location location, Long timestamp,
                                                               Callable<MetadataResult<MetadataKeysContainer>> callable) {
    Pair<MetadataCache, Map<String, String>> input = getDataSenseCache(location);
    String result =
        (String) input.getFirst()
            .getMetadataKeys(new ImmutableMetadataCacheKeyInfo(componentId, toLocationDTO(location).toString(), timestamp,
                                                               input.getSecond()),
                             () -> {
                               final MetadataResult<MetadataKeysContainer> runtimeMetadataResult = callable.call();
                               List<MetadataFailure> failures = emptyList();
                               if (!runtimeMetadataResult.isSuccess()) {
                                 failures = toMetadataFailuresDTO(runtimeMetadataResult.getFailures());
                               }
                               return new ImmutableMetadataResult(runtimeMetadataResult.isSuccess(),
                                                                  serialize(runtimeMetadataResult), failures);
                             });

    return (MetadataResult<MetadataKeysContainer>) deserialize(result);
  }

  private Pair<MetadataCache, Map<String, String>> getDataSenseCache(Location location) {
    Pair<MetadataCache, Map<String, String>> input;
    if (dataSenseArtifact.getParent().isPresent() && dataSenseArtifact.getParent().get().hasComponentModel(location)) {
      DataSenseArtifact parent = this.dataSenseArtifact.getParent().get();
      input = new Pair<>(parent.getMetadataCache(), parent.getProperties());
    } else {
      input = new Pair<>(dataSenseArtifact.getMetadataCache(), dataSenseArtifact.getProperties());
    }
    return input;
  }

}
