/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.artifact;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.tooling.client.api.artifact.resources.ResourceLoader;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.application.Artifact;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.io.InputStream;
import java.net.URL;
import java.util.Optional;

/**
 * Default implementation of {@link ResourceLoader}.
 */
public class DefaultResourceLoader implements ResourceLoader, Command {

  private ResourceLoaderFormatter formatter = new ResourceLoaderFormatter();

  private Artifact artifact;
  private Serializer serializer;

  public DefaultResourceLoader(Artifact artifact, Serializer serializer) {
    this.artifact = artifact;
    this.serializer = serializer;
  }

  @Override
  public InputStream getResourceAsStream(String resource) {
    return getClassLoader().getResourceAsStream(resource);
  }

  @Override
  public Optional<URL> findIn(String resource, String groupId, String artifactId, Optional<String> version, String classifier,
                              String type) {
    return ofNullable(getResource(formatter.format(resource, groupId, artifactId, version, classifier, type)));
  }

  @Override
  public Optional<InputStream> loadFrom(String resource, String groupId, String artifactId, Optional<String> version,
                                        String classifier, String type) {
    return ofNullable(getResourceAsStream(formatter.format(resource, groupId, artifactId, version, classifier, type)));
  }

  private ClassLoader getClassLoader() {
    return artifact.getArtifactClassLoader().getRegionClassLoader().getClassLoader();
  }

  @Override
  public URL getResource(String resource) {
    return getClassLoader().getResource(resource);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getResourceAsStream": {
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return getResourceAsStream(arguments[0]);
      }
      case "getResource": {
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return getResource(arguments[0]);
      }
      case "findIn": {
        checkState(classes.length == 6 && classes[0].equals(String.class.getName()) && classes[1].equals(String.class.getName())
            &&
            classes[2].equals(String.class.getName()) && classes[3].equals(Optional.class.getName())
            && classes[4].equals(String.class.getName()) && classes[5].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return findIn(arguments[0], arguments[1], arguments[2], serializer.deserialize(arguments[3]), arguments[4], arguments[5]);
      }
      case "loadFrom": {
        checkState(classes.length == 6 && classes[0].equals(String.class.getName()) && classes[1].equals(String.class.getName())
            &&
            classes[2].equals(String.class.getName()) && classes[3].equals(Optional.class.getName())
            && classes[4].equals(String.class.getName()) && classes[5].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return loadFrom(arguments[0], arguments[1], arguments[2], serializer.deserialize(arguments[3]), arguments[4],
                        arguments[5]);
      }

    }
    throw methodNotFound(this.getClass(), methodName);
  }

  /**
   * Helper class to build the resource loader formatter string.
   */
  public static class ResourceLoaderFormatter {

    private static final String RESOURCE_FORMAT = "resource::%s:%s:%s:%s:%s:%s";

    public String format(String resource, String groupId, String artifactId, Optional<String> version, String classifier,
                         String type) {
      return String.format(RESOURCE_FORMAT, groupId, artifactId, version.orElse(""), classifier, type, resource);
    }

  }

}
