/*
/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import static java.util.Collections.emptyMap;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static org.mule.runtime.api.exception.ExceptionHelper.getRootException;
import static org.mule.runtime.api.metadata.DataType.fromFunction;
import static org.mule.runtime.api.metadata.DataType.fromType;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse.builder;
import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.BindingContextUtils;
import org.mule.runtime.api.el.ExpressionFunction;
import org.mule.runtime.api.message.ItemSequenceInfo;
import org.mule.runtime.api.meta.NamedObject;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.security.Authentication;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.util.StringUtils;
import org.mule.runtime.core.internal.el.mvel.function.PropertyAccessFunction;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.weave.v2.el.WeaveExpressionLanguage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import org.apache.commons.io.IOUtils;

/**
 * {@link DataWeaveRunner} that executes an script locally using DataWeave API.
 *
 *
 * @since 4.0
 */
public class LocalRunner implements DataWeaveRunner {

  private static final String APPLICATION_JAVA = "application/java";
  private static final String DEFAULT_ATTRIBUTES_MEDIA_TYPE = "application/java";

  private static final DataType VARS_DATA_TYPE = DataType.builder()
      .mapType(Map.class)
      .keyType(String.class)
      .valueType(TypedValue.class)
      .build();

  public final static TypedValue EMPTY_VARS = new TypedValue<>(emptyMap(), VARS_DATA_TYPE);
  private final static Supplier<TypedValue> EMPTY_VARS_SUPPLIER = () -> EMPTY_VARS;
  public static final TypedValue NULL_TYPED_VALUE = new TypedValue<>(null, DataType.OBJECT);
  private static final Supplier<TypedValue> NULL_TYPED_VALUE_SUPPLIER = () -> NULL_TYPED_VALUE;

  private LazyValue<ClassLoader> executionClassLoader;
  private LazyValue<ConfigurationProperties> configurationProperties;

  /**
   * Creates an instance of the runner with the given {@link ClassLoader}.
   *
   * @param executionClassLoader the {@link ClassLoader} to be used when running DW scripts.
   */
  public LocalRunner(LazyValue<ClassLoader> executionClassLoader, LazyValue<ConfigurationProperties> configurationProperties) {
    requireNonNull(executionClassLoader, "executionClassLoader cannot be null");
    requireNonNull(configurationProperties, "configurationProperties cannot be null");

    this.executionClassLoader = executionClassLoader;
    this.configurationProperties = configurationProperties;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    return withContextClassLoader(executionClassLoader.get(), () -> {
      try {
        WeaveExpressionLanguage executor = new WeaveExpressionLanguage();
        TypedValue response =
            executor.evaluate(getScript(parameters), buildContextFromParameters(parameters), parameters.getRequestTimeout());
        return builder().result(getResult(response)).mediaType(getMediaType(response)).valid(true).build();
      } catch (Exception e) {
        return builder().errorMessage(getRootException(e).getMessage()).build();
      }
    });
  }

  private String getMediaType(TypedValue evaluate) {
    DataType dataType = evaluate.getDataType();
    MediaType mediaType = dataType.getMediaType();
    return mediaType.withoutParameters().toString();
  }

  private String getResult(TypedValue result) {
    final Object value = result.getValue();
    if (value instanceof CursorStreamProvider) {
      return manageCursorStream(result);
    } else {
      return serializeObject(value);
    }
  }

  protected String manageCursorStream(TypedValue<CursorStreamProvider> result) {
    CursorStreamProvider streamProvider = null;
    try {
      streamProvider = result.getValue();
      Charset encoding = result.getDataType().getMediaType().getCharset().orElseGet(Charset::defaultCharset);
      try (CursorStream cursorStream = streamProvider.openCursor()) {
        return IOUtils.toString(cursorStream, encoding);
      }
    } catch (IOException e) {
      throw new ToolingException("DataWeave :: Error consuming DataWeave result", e);
    } finally {
      if (streamProvider != null) {
        streamProvider.close();
      }
    }
  }

  protected String serializeObject(Object value) {
    Gson gson = new GsonBuilder().setPrettyPrinting().create();
    return gson.toJson(value);
  }

  private BindingContext buildContextFromParameters(DataWeavePreviewRequest parameters) throws IOException {
    BindingContext.Builder bindingBuilder = BindingContext.builder();

    final EventModel event = getEvent(parameters);
    if (event != null) {
      manageVariables(bindingBuilder, event);
      manageMessage(bindingBuilder, event);
    } else {
      bindingBuilder.addBinding(BindingContextUtils.ATTRIBUTES, NULL_TYPED_VALUE_SUPPLIER);
      bindingBuilder.addBinding(BindingContextUtils.PAYLOAD, NULL_TYPED_VALUE_SUPPLIER);
      bindingBuilder.addBinding(BindingContextUtils.DATA_TYPE, NULL_TYPED_VALUE_SUPPLIER);
      bindingBuilder.addBinding(BindingContextUtils.VARS, EMPTY_VARS_SUPPLIER);
    }

    appendGlobalBindings(bindingBuilder);

    appendFunctions(bindingBuilder);

    return bindingBuilder.build();
  }

  private void appendGlobalBindings(BindingContext.Builder bindingBuilder) {
    bindingBuilder.addBinding(BindingContextUtils.AUTHENTICATION,
                              new TypedValue(null, DataType.fromType(Authentication.class)));
    bindingBuilder.addBinding(BindingContextUtils.CORRELATION_ID, new TypedValue(null, DataType.STRING));
    bindingBuilder.addBinding(BindingContextUtils.FLOW, new TypedValue(null, DataType.fromType(NamedObject.class)));
    bindingBuilder.addBinding(BindingContextUtils.ITEM_SEQUENCE_INFO,
                              new TypedValue(null, DataType.fromType(ItemSequenceInfo.class)));
    bindingBuilder.addBinding(BindingContextUtils.ERROR, new TypedValue(null, DataType.fromType(Error.class)));
  }

  private void appendFunctions(BindingContext.Builder builder) {
    PropertyAccessFunction propertyFunction = new PropertyAccessFunction(new ConfigurationProperties() {

      @Override
      public <T> Optional<T> resolveProperty(String propertyKey) {
        return configurationProperties.get().resolveProperty(propertyKey);
      }

      @Override
      public Optional<Boolean> resolveBooleanProperty(String property) {
        return configurationProperties.get().resolveBooleanProperty(property);
      }

      @Override
      public Optional<String> resolveStringProperty(String property) {
        Optional<String> stringProperty = configurationProperties.get().resolveStringProperty(property);
        if (stringProperty.isPresent() && StringUtils.isEmpty(stringProperty.get())) {
          return empty();
        }
        return stringProperty;
      }
    });
    builder.addBinding("p", new TypedValue(propertyFunction, fromFunction(propertyFunction)));

    ExpressionFunction lookupFunction = new MockLookupFunction();
    builder.addBinding("lookup", new TypedValue(lookupFunction, fromFunction(lookupFunction)));

    ExpressionFunction causedByFunction = new MockCausedByFunction();
    builder.addBinding("causedBy", new TypedValue(causedByFunction, fromFunction(causedByFunction)));

  }

  private void manageMessage(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    final MessageModel message = event.getMessage();
    if (message != null) {
      if (message.getAttributes() != null) {
        bindingBuilder.addBinding(BindingContextUtils.ATTRIBUTES, asTypedValue(message.getAttributes()));
      } else {
        bindingBuilder.addBinding(BindingContextUtils.ATTRIBUTES, NULL_TYPED_VALUE_SUPPLIER);
      }

      if (message.getPayload() != null) {
        bindingBuilder.addBinding(BindingContextUtils.PAYLOAD, asTypedValue(message.getPayload()));
        bindingBuilder.addBinding(BindingContextUtils.DATA_TYPE, new org.mule.runtime.api.metadata.TypedValue(DataType.builder()
            .type(String.class)
            .mediaType(message.getPayload().getDataType().getMediaType())
            .build(),
                                                                                                              fromType(DataType.class)));
      } else {
        bindingBuilder.addBinding(BindingContextUtils.PAYLOAD, NULL_TYPED_VALUE_SUPPLIER);
        bindingBuilder.addBinding(BindingContextUtils.DATA_TYPE, NULL_TYPED_VALUE_SUPPLIER);
      }
    } else {
      bindingBuilder.addBinding(BindingContextUtils.ATTRIBUTES, NULL_TYPED_VALUE_SUPPLIER);
      bindingBuilder.addBinding(BindingContextUtils.PAYLOAD, NULL_TYPED_VALUE_SUPPLIER);
      bindingBuilder.addBinding(BindingContextUtils.DATA_TYPE, NULL_TYPED_VALUE_SUPPLIER);
    }
  }

  private void manageVariables(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    if (event.getVariables() != null && !event.getVariables().isEmpty()) {
      Map<String, TypedValue> variables = new HashMap<>();
      for (Map.Entry<String, TypedValueModel> pair : event.getVariables().entrySet()) {
        variables.put(pair.getKey(), asTypedValue(pair.getValue()));
      }
      DataType variablesDataType = DataType.builder().type(Map.class).mediaType(DEFAULT_ATTRIBUTES_MEDIA_TYPE).build();
      bindingBuilder.addBinding(BindingContextUtils.VARS, new TypedValue<>(variables, variablesDataType));
    } else {
      bindingBuilder.addBinding(BindingContextUtils.VARS, EMPTY_VARS);
    }
  }

  protected TypedValue<?> asTypedValue(TypedValueModel restTypedValue) throws IOException {
    String mediaType = restTypedValue.getDataType().getMediaType();
    if (APPLICATION_JAVA.equals(mediaType)) {
      throw new IllegalArgumentException("Java input not supported, serialize to DW script");
    } else {
      DataType dataType = DataType.builder().type(String.class).mediaType(mediaType).build();
      return new TypedValue<>(new String(restTypedValue.getContent(), "UTF-8"), dataType);
    }
  }

  private EventModel getEvent(DataWeavePreviewRequest parameters) {
    return parameters.getEvent();
  }

  private String getScript(DataWeavePreviewRequest parameters) {
    return parameters.getScript();
  }
}
