/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.lang.String.format;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.deployment.model.api.domain.DomainDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.DefaultApplicationModelFactory;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import com.google.common.collect.ImmutableSet;

import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Represents a Mule domain and allows to be deployed into Mule Runtime.
 *
 * @since 4.0
 */
public class DefaultDomain extends AbstractArtifact<DomainDescriptor> implements Domain {

  private ReentrantLock lock = new ReentrantLock(true);

  public DefaultDomain(String id, ArtifactResources artifactResources, DomainDescriptor domainDescriptor,
                       ToolingArtifactContext context,
                       Map<String, String> properties) {
    super(id, artifactResources, domainDescriptor, context, properties);
  }

  @Override
  public DomainDescriptor getDescriptor() {
    return artifactDescriptor;
  }

  @Override
  protected LazyValue<ToolingArtifactClassLoader> newToolingArtifactClassLoaderLazyValue() {
    return new LazyValue<>(() -> context.getDomainClassLoaderFactory().createDomainClassLoader(artifactDescriptor));
  }

  @Override
  protected LazyValue<ToolingApplicationModel> newToolingApplicationModelLazyValue() {
    return new LazyValue<>(() -> new ToolingApplicationModel(null,
                                                             new DefaultApplicationModelFactory()
                                                                 .createApplicationModel(artifactDescriptor,
                                                                                         ImmutableSet.<ExtensionModel>builder()
                                                                                             .addAll(getExtensionModels())
                                                                                             .build(),
                                                                                         getArtifactClassLoader()
                                                                                             .getClassLoader(),
                                                                                         noFailureConfigurationProperties,
                                                                                         getProperties())
                                                                 .orElseThrow(() -> new ToolingException(format("Couldn't create ApplicationModel from %s",
                                                                                                                this))),
                                                             getArtifactClassLoader().getClassLoader()));
  }

  @Override
  public ReentrantLock getRemoteInvokerLock() {
    return lock;
  }

  @Override
  public <R> R evaluateWithRemoteDomain(DomainRemoteFunction<R> function) {
    checkState();
    try {
      getRemoteInvokerLock().lock();
      if (remoteArtifactId == null) {
        remoteArtifactId = artifactDeployer.deploy(null);
      }
      return function.apply(remoteArtifactId, runtimeToolingService.get());
    } catch (NoSuchApplicationException e) {
      remoteArtifactId = artifactDeployer.deploy(null);
      return function.apply(remoteArtifactId, runtimeToolingService.get());
    } finally {
      getRemoteInvokerLock().unlock();
    }
  }

}
