/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static javax.tools.ToolProvider.getSystemJavaCompiler;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;

import com.esotericsoftware.minlog.Log;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.List;

import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.StandardJavaFileManager;

import org.apache.commons.io.FileUtils;
import org.jetbrains.annotations.Nullable;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExtendedCompatibleFieldSerializerTestCase {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private URLClassLoader classLoaderV1;
  private URLClassLoader classLoaderV2;

  @Before
  public void before() throws IOException {
    File targetFolderV1 = temporaryFolder.newFolder();
    File targetFolderV2 = temporaryFolder.newFolder();

    compile(targetFolderV1, getSourceFile("sources/v1/Pojo.java"), getSourceFile("sources/v1/Bar.java"),
            getSourceFile("sources/v1/Foo.java"));
    compile(targetFolderV2, getSourceFile("sources/v2/Pojo.java"), getSourceFile("sources/v2/Bar.java"),
            getSourceFile("sources/v2/Foo.java"), getSourceFile("sources/v2/Zee.java"));

    classLoaderV1 = new URLClassLoader(new URL[] {targetFolderV1.toURI().toURL()});
    classLoaderV2 = new URLClassLoader(new URL[] {targetFolderV2.toURI().toURL()});
  }

  @Nullable
  public File getSourceFile(String resource) {
    return FileUtils.toFile(this.getClass().getClassLoader().getResource(resource));
  }

  public void compile(File targetFolder, File... sourceFiles) {
    final JavaCompiler compiler = getSystemJavaCompiler();
    final StandardJavaFileManager fileManager = compiler.getStandardFileManager(null, null, null);
    try {

      Iterable<? extends JavaFileObject> compilationUnits = fileManager.getJavaFileObjects(sourceFiles);

      List<String> options = new ArrayList<>();
      options.add("-d");
      options.add(targetFolder.getAbsolutePath());

      JavaCompiler.CompilationTask task = compiler.getTask(null, fileManager, null, options, null, compilationUnits);

      Boolean status = task.call();
      if (!status) {
        throw new RuntimeException("Compiler task finished with error. Enable logging to find more information");
      }
    } catch (Throwable e) {
      throw e;
    } finally {
      try {
        fileManager.close();
      } catch (IOException e) {
        // Ignore
      }
    }
  }

  @Test
  public void removeFieldsWhenDeserializingServerSide() throws Exception {
    if (logger.isTraceEnabled()) {
      Log.set(Log.LEVEL_TRACE);
    }
    File localMavenRepository = new File("/tmp");

    KryoClientSerializer serializer = new KryoClientSerializer(classLoaderV2, classLoaderV1);
    Object pojoV2 = classLoaderV2.loadClass("Pojo").newInstance();
    setField(pojoV2, "localMavenRepositoryLocation", localMavenRepository);

    String serialization = serializer.safeSerialize(pojoV2);

    KryoServerSerializer deserializer = new KryoServerSerializer(classLoaderV1, classLoaderV2);
    Class pojoV1Clazz = classLoaderV1.loadClass("Pojo");
    Object pojoV1 = pojoV1Clazz.newInstance();
    setField(pojoV1, "localMavenRepositoryLocation", localMavenRepository);
    assertThat(pojoV1, equalTo(deserializer.deserialize(serialization, pojoV1Clazz)));
  }

  @Test
  public void removeFieldsWhenDeserializingClientSide() throws Exception {
    if (logger.isTraceEnabled()) {
      Log.set(Log.LEVEL_TRACE);
    }
    File localMavenRepository = new File("/tmp");

    KryoServerSerializer serializer = new KryoServerSerializer(classLoaderV1, classLoaderV2);
    Object pojoV1 = classLoaderV1.loadClass("Pojo").newInstance();
    setField(pojoV1, "localMavenRepositoryLocation", localMavenRepository);

    String serialization = serializer.safeSerialize(pojoV1);

    KryoServerSerializer deserializer = new KryoServerSerializer(classLoaderV2, classLoaderV1);
    Class pojoV2Clazz = classLoaderV2.loadClass("Pojo");
    Object pojoV2 = pojoV2Clazz.newInstance();
    setField(pojoV2, "localMavenRepositoryLocation", localMavenRepository);
    assertThat(pojoV2, equalTo(deserializer.deserialize(serialization, pojoV2Clazz)));
  }

  @Test
  public void missingClassForFieldOnTargetClassLoader() throws Exception {
    if (logger.isTraceEnabled()) {
      Log.set(Log.LEVEL_TRACE);
    }
    KryoClientSerializer serializer = new KryoClientSerializer(classLoaderV2, classLoaderV1);
    Object fooV2 = classLoaderV2.loadClass("Foo").newInstance();

    String serialization = serializer.safeSerialize(fooV2);

    KryoServerSerializer deserializer = new KryoServerSerializer(classLoaderV1, classLoaderV2);
    Class fooV1Clazz = classLoaderV1.loadClass("Foo");
    Object fooV1 = fooV1Clazz.newInstance();
    assertThat(fooV1, equalTo(deserializer.deserialize(serialization, fooV1Clazz)));
  }

  private void setField(Object pojoV2, String fieldName, Object value) throws NoSuchFieldException, IllegalAccessException {
    Field localMavenRepositoryLocations = pojoV2.getClass().getDeclaredField(fieldName);
    localMavenRepositoryLocations.setAccessible(true);
    localMavenRepositoryLocations.set(pojoV2, value);
  }

}
