/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Collections.sort;
import static java.util.Comparator.comparingLong;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.runtime.api.util.LazyValue;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationName.messageHistory.AgentTrackingNotificationResponse;
import org.mule.tooling.client.api.message.history.MessageHistory;
import org.mule.tooling.client.api.message.history.MessageHistoryService;
import org.mule.tooling.client.api.types.Transaction;
import org.mule.tooling.client.api.types.TransactionStackEntry;
import org.mule.tooling.client.api.types.TransactionStatus;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation for {@link MessageHistoryService}.
 *
 * @since 4.0
 */
public class DefaultMessageHistoryService implements MessageHistoryService, Command {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String PROCESSORS = "processors";
  private static final String FIRST_MESSAGE_PROCESSOR_INDEX = "/0";
  private static final String MESSAGE_PRE_INVOKE_ACTION = "message processor pre invoke";
  private static final String MESSAGE_POST_INVOKE_ACTION = "message processor post invoke";
  private static final String MESSAGE_EXCEPTION = "exception";

  private LazyValue<RuntimeToolingService> runtimeToolingServiceLazyValue;
  private Serializer serializer;

  /**
   * Creates a default instance of the service.
   *
   * @param runtimeToolingServiceLazyValue {@link LazyValue} for {@link RuntimeToolingService}.
   */
  protected DefaultMessageHistoryService(LazyValue<RuntimeToolingService> runtimeToolingServiceLazyValue, Serializer serializer) {
    requireNonNull(runtimeToolingServiceLazyValue, "runtimeToolingServiceLazyValue cannot be null");
    requireNonNull(serializer, "serializer cannot be null");

    this.runtimeToolingServiceLazyValue = runtimeToolingServiceLazyValue;
    this.serializer = serializer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void enable(String applicationName) {
    runtimeToolingServiceLazyValue.get().enableMessageHistory(applicationName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageHistory consume(String applicationName, int chunkSize) {
    List<AgentTrackingNotificationResponse> agentTrackingNotifications =
        runtimeToolingServiceLazyValue.get().consumeMessageHistoryNotifications(applicationName, chunkSize);
    return convertToMessageHistory(agentTrackingNotifications);
  }

  private MessageHistory convertToMessageHistory(List<AgentTrackingNotificationResponse> agentTrackingNotifications) {
    MessageHistory messageHistory = new MessageHistory();
    List<Transaction> transactions = new ArrayList<>();
    messageHistory.setTransactions(transactions);
    Map<String, List<AgentTrackingNotificationResponse>> notificationsMap =
        splitAgentNotificationsAndSortByDate(agentTrackingNotifications);
    notificationsMap.keySet().stream().forEach(transactionId -> {
      List<AgentTrackingNotificationResponse> transactionNotifications = notificationsMap.get(transactionId);
      Transaction transaction = new Transaction();
      // TODO MULE-12229 Support TransactionStatus
      transaction.setTransactionStatus(TransactionStatus.COMPLETE);
      transaction.setId(transactionNotifications.get(0).getTransactionId());
      AgentTrackingNotificationResponse firstMessageProcessorInputNotification = transactionNotifications.get(0);
      if (firstMessageProcessorInputNotification.getComponentLocation().getLocation().endsWith(FIRST_MESSAGE_PROCESSOR_INDEX) &&
          firstMessageProcessorInputNotification.getAction().equals(MESSAGE_PRE_INVOKE_ACTION)) {
        transaction.setMessage(firstMessageProcessorInputNotification.getEvent().getMessage());
        transaction.setTimestamp(firstMessageProcessorInputNotification.getTimestamp());
        transaction.setGlobalName(getGlobalNameFromLocation(firstMessageProcessorInputNotification.getComponentLocation()
            .getLocation()));
      }

      transaction.setTransactionStack(transactionNotifications.stream()
          .filter(agentTrackingNotification -> agentTrackingNotification.getAction().equals(MESSAGE_POST_INVOKE_ACTION))
          .map(agentTrackingNotification -> {
            TransactionStackEntry transactionStackEntry = new TransactionStackEntry();
            transactionStackEntry.setTimestamp(agentTrackingNotification.getTimestamp());
            transactionStackEntry.setComponentLocation(agentTrackingNotification.getComponentLocation());
            transactionStackEntry.setEventModel(agentTrackingNotification.getEvent());
            setInputEvent(transactionNotifications, agentTrackingNotification, transactionStackEntry);

            return transactionStackEntry;
          }).collect(toList()));
      transactions.add(transaction);
    });
    return messageHistory;
  }

  private void setInputEvent(List<AgentTrackingNotificationResponse> transactionNotifications,
                             AgentTrackingNotificationResponse currentAgentTrackingNotificationResponse,
                             TransactionStackEntry currentTransactionStackEntry) {
    if (currentTransactionStackEntry.getComponentLocation().getLocation().endsWith(PROCESSORS + FIRST_MESSAGE_PROCESSOR_INDEX)) {
      final ListIterator<AgentTrackingNotificationResponse> listIterator =
          transactionNotifications.listIterator(transactionNotifications.indexOf(currentAgentTrackingNotificationResponse));
      while (listIterator.hasPrevious() && currentTransactionStackEntry.getInput() == null) {
        AgentTrackingNotificationResponse previous = listIterator.previous();
        if (previous.getComponentLocation().equals(currentAgentTrackingNotificationResponse.getComponentLocation())
            && previous.getAction().equals(MESSAGE_PRE_INVOKE_ACTION)) {
          currentTransactionStackEntry.setInput(previous.getEvent());
        }
      }
    }
  }

  private Map<String, List<AgentTrackingNotificationResponse>> splitAgentNotificationsAndSortByDate(List<AgentTrackingNotificationResponse> agentTrackingNotifications) {
    logger.debug("Grouping notifications by correlationId");
    Map<String, List<AgentTrackingNotificationResponse>> notificationsMap = new HashMap<>();
    agentTrackingNotifications.stream().forEach(agentTrackingNotification -> {
      logger.debug("Processing notification: {}", agentTrackingNotification);
      List<AgentTrackingNotificationResponse> notifications = notificationsMap.get(agentTrackingNotification.getCorrelationId());
      if (notifications == null) {
        notifications = new ArrayList<>();
        notificationsMap.put(agentTrackingNotification.getCorrelationId(), notifications);
      }
      notifications.add(agentTrackingNotification);
    });

    for (List<AgentTrackingNotificationResponse> transactinAgentNotifications : notificationsMap.values()) {
      sort(transactinAgentNotifications, comparingLong(AgentTrackingNotificationResponse::getTimestamp));
    }
    return notificationsMap;
  }

  private static String getGlobalNameFromLocation(String location) {
    return org.mule.tooling.client.api.component.location.Location.builderFromStringRepresentation(location).build()
        .getGlobalName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void disable(String applicationName) {
    try {
      runtimeToolingServiceLazyValue.get().disableMessageHistory(applicationName);
    } catch (Exception e) {
      logger.warn("Error while disabling application for try it", e);
    }
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "enable": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        enable(serializer.deserialize(arguments[0]));
        return null;
      }
      case "consume": {
        checkState(arguments.length == 2,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 2 && classes[0].equals(String.class.getName()) && classes[1].equals(Integer.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(consume(serializer.deserialize(arguments[0]), serializer.deserialize(arguments[1])));
      }
      case "disable": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        disable(serializer.deserialize(arguments[0]));
        return null;
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
