/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Objects.requireNonNull;
import static org.mule.tooling.client.internal.serialization.XStreamFactory.createXStream;
import org.mule.tooling.client.api.datasense.ImmutableMetadataCacheKeyInfo;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.serialization.XStreamFactory;

import com.thoughtworks.xstream.XStream;

import java.io.Serializable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.Callable;

import org.slf4j.MDC;

/**
 * A proxy implementation that uses reflection to call a target instance implementation loaded by a different class loader.
 *
 * @since 4.0
 */
public class MetadataCacheProxy implements MetadataCache {

  private Object proxyTarget;
  private XStream xStream;
  private Object proxyXstream;

  public MetadataCacheProxy(Object proxyTarget) {
    requireNonNull(proxyTarget, "proxyTarget cannot be null");

    this.proxyTarget = proxyTarget;
    this.xStream = createXStream();
    this.xStream.ignoreUnknownElements();
    try {
      this.proxyXstream = proxyTarget.getClass().getClassLoader().loadClass(XStreamFactory.class.getName())
          .getMethod("createXStream").invoke(null);
      this.proxyXstream.getClass().getMethod("ignoreUnknownElements").invoke(this.proxyXstream);
    } catch (Exception e) {
      throw new RuntimeException("Error while creating proxy for XStream with clients class loader", e);
    }
  }

  public Serializable invoke(String methodName, String componentId, String location, Long timestamp,
                             Map<String, String> toolingArtifactProperties,
                             Callable callable) {
    try {
      Method method =
          proxyTarget.getClass()
              .getMethod(methodName, proxyTarget.getClass().getClassLoader().loadClass(MetadataCacheKeyInfo.class.getName()),
                         Callable.class);
      method.setAccessible(true);
      Callable proxyCallable = new CallableProxy(callable);
      return (Serializable) method.invoke(proxyTarget,
                                          new Object[] {
                                              createMetadataCacheKeyInfo(componentId, location, timestamp,
                                                                         toolingArtifactProperties),
                                              proxyCallable});
    } catch (Exception e) {
      throw new ToolingException("Error while calling proxy method on client code", e);
    }
  }


  private class CallableProxy implements Callable {

    private Callable proxyTarget;

    public CallableProxy(Callable proxyTarget) {
      this.proxyTarget = proxyTarget;
    }

    @Override
    public Object call() throws Exception {
      final Object result = proxyTarget.call();
      final String xml = xStream.toXML(result);
      return proxyXstream.getClass().getMethod("fromXML", new Class[] {String.class}).invoke(proxyXstream, xml);
    }
  }

  private Object createMetadataCacheKeyInfo(String componentId, String location, Long timestamp,
                                            Map<String, String> toolingArtifactProperties)
      throws ClassNotFoundException, NoSuchMethodException, IllegalAccessException, InvocationTargetException,
      InstantiationException {
    final ClassLoader classLoader = proxyTarget.getClass().getClassLoader();
    // At this point the MDC is correctly set but as the implementation to be invoked has
    // been loaded by another class loader (client's class loader) so we have to propagate the MDC
    // context using reflection. This class is loaded by an isolated class loader therefore this propagation has
    // to be made.
    classLoader.loadClass(MDC.class.getName()).getMethod("setContextMap", Map.class).invoke(null, MDC.getCopyOfContextMap());
    return classLoader.loadClass(ImmutableMetadataCacheKeyInfo.class.getName())
        .getConstructor(String.class, String.class, Long.class, Map.class)
        .newInstance(componentId, location, timestamp, toolingArtifactProperties);
  }

  @Override
  public Serializable getOperationMetadata(MetadataCacheKeyInfo metadataCacheKeyInfo,
                                           Callable resolver) {
    return invoke("getOperationMetadata", metadataCacheKeyInfo.getComponentId(), metadataCacheKeyInfo.getLocation(),
                  metadataCacheKeyInfo.getTimestamp(), metadataCacheKeyInfo.getToolingArtifactProperties(), resolver);
  }

  @Override
  public Serializable getSourceMetadata(MetadataCacheKeyInfo metadataCacheKeyInfo,
                                        Callable<MetadataResult> resolver) {
    return invoke("getSourceMetadata", metadataCacheKeyInfo.getComponentId(), metadataCacheKeyInfo.getLocation(),
                  metadataCacheKeyInfo.getTimestamp(), metadataCacheKeyInfo.getToolingArtifactProperties(), resolver);
  }

  @Override
  public Serializable getMetadataKeys(MetadataCacheKeyInfo metadataCacheKeyInfo,
                                      Callable<MetadataResult> resolver) {
    return invoke("getMetadataKeys", metadataCacheKeyInfo.getComponentId(), metadataCacheKeyInfo.getLocation(),
                  metadataCacheKeyInfo.getTimestamp(), metadataCacheKeyInfo.getToolingArtifactProperties(), resolver);
  }

}
