/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toSet;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.resources.ResourceLoader;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingService;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.Artifact;
import org.mule.tooling.client.internal.application.Domain;
import org.mule.tooling.client.internal.artifact.DefaultResourceLoader;
import org.mule.tooling.client.internal.datasense.DataSenseArtifact;
import org.mule.tooling.client.internal.dataweave.*;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.service.ServiceRegistry;

import java.util.Map;
import java.util.Optional;

/**
 * Implementation that allows to do Tooling operations without deploying the application until it is needed.
 * <p/>
 * If an operation requires the application to be deployed it will deploy it first and then delegate the operation to the default
 * implementation of {@link ToolingArtifact}.
 *
 * @since 4.0
 */
public class DefaultToolingArtifact implements ToolingArtifact, Command {

  private String id;
  private Artifact artifact;
  private Serializer serializer;

  private ConnectivityTestingService connectivityTestingService;
  private MetadataService metadataService;
  private DataSenseService dataSenseService;
  private DataWeaveService dataWeaveService;
  private ValueProviderService valueProviderService;

  private ToolingArtifact parentToolingArtifact;

  /**
   * Creates an instance of the {@link DefaultToolingArtifact} from a fetched applicationId or deploys the application to obtain
   * an identifier in case if null.
   *
   * @param id {@link String} identifier for this {@link ToolingArtifact}. Non null.
   * @param metadataCache {@link MetadataCache} to be used by this {@link ToolingArtifact}. Non null.
   * @param application {@link Application} to handle state, local and remote (Mule Runtime) in order to resolve operations. Non null.
   * @param parentMetadataCache {@link MetadataCache} to be used by the parent {@link ToolingArtifact}. Can be null.
   * @param serializer {@link Serializer} to serialize parameters and results when called from Tooling API. Non null.
   */
  public DefaultToolingArtifact(String id, LazyValue<MetadataCache> metadataCache, Application application,
                                ToolingArtifact parentToolingArtifact, LazyValue<MetadataCache> parentMetadataCache,
                                Serializer serializer, ServiceRegistry serviceRegistry) {
    checkNotNull(id, "id cannot be null");
    checkNotNull(application, "application cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");
    checkNotNull(serializer, "serializer cannot be null");
    checkNotNull(serviceRegistry, "serviceRegistry cannot be null");

    this.id = id;
    this.artifact = application;
    this.parentToolingArtifact = parentToolingArtifact;
    this.serializer = serializer;

    // Services registration...
    DataWeaveRunner remoteRunner = null;
    DataWeaveRunner localRunner = new LocalRunner(new LazyValue(() -> application.getArtifactClassLoader().getClassLoader()),
                                                  new LazyValue(() -> application.getApplicationModel().getMuleApplicationModel()
                                                      .getConfigurationProperties()));

    MetadataProvider metadataProvider = new InternalApplicationMetadataProvider(application);

    remoteRunner = new ApplicationRemoteRunner(application);
    this.connectivityTestingService = new ApplicationConnectivityTestingService(application, serializer);
    this.metadataService =
        new ToolingMetadataServiceAdapter(new LazyValue<>(() -> application.getApplicationModel().getMuleApplicationModel()),
                                          new LazyValue<>(() -> application.getExtensionModels().stream().collect(toSet())),
                                          metadataProvider, metadataCache, application.getProperties(), serializer);

    DataSenseArtifact parentDataSenseArtifact = application.getDomain()
        .map(domain -> new DataSenseArtifact(domain, parentMetadataCache, new InternalDomainMetadataProvider(domain),
                                             domain.getProperties(), null))
        .orElse(null);
    DataSenseArtifact dataSenseArtifact =
        new DataSenseArtifact(application, metadataCache, metadataProvider, application.getProperties(),
                              parentDataSenseArtifact);


    this.dataSenseService = new DefaultDataSenseService(dataSenseArtifact, serializer, serviceRegistry);
    this.valueProviderService = new ApplicationValueProviderService(application, serializer);

    this.dataWeaveService =
        new DefaultDataWeaveService(new LazyValue(() -> application.getArtifactClassLoader().getClassLoader()),
                                    new DataWeaveRunnerProvider(localRunner, remoteRunner),
                                    new ModulesAnalyzer(), serializer);
  }

  public DefaultToolingArtifact(String id, LazyValue<MetadataCache> metadataCache, Domain domain, Serializer serializer,
                                ServiceRegistry serviceRegistry) {
    checkNotNull(id, "id cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");
    checkNotNull(domain, "domain cannot be null");
    checkNotNull(serializer, "serializer cannot be null");
    checkNotNull(serviceRegistry, "serviceRegistry cannot be null");

    this.id = id;
    this.artifact = domain;
    this.serializer = serializer;

    // Services registration...
    DataWeaveRunner remoteRunner = null;
    DataWeaveRunner localRunner = new LocalRunner(new LazyValue(() -> domain.getArtifactClassLoader().getClassLoader()),
                                                  new LazyValue(() -> domain.getApplicationModel().getMuleApplicationModel()
                                                      .getConfigurationProperties()));

    MetadataProvider metadataProvider = new InternalDomainMetadataProvider(domain);
    remoteRunner = new DomainRemoteRunner(domain);
    this.connectivityTestingService = new DomainConnectivityTestingService(domain, serializer);
    this.dataSenseService = new UnavailableDataSenseService(ArtifactType.DOMAIN, serializer, serviceRegistry);
    this.metadataService =
        new ToolingMetadataServiceAdapter(new LazyValue<>(() -> domain.getApplicationModel().getMuleApplicationModel()),
                                          new LazyValue<>(() -> domain.getExtensionModels().stream().collect(toSet())),
                                          metadataProvider, metadataCache, domain.getProperties(), serializer);
    this.valueProviderService = new DomainValueProviderService(domain, serializer);

    this.dataWeaveService =
        new DefaultDataWeaveService(new LazyValue(() -> domain.getArtifactClassLoader().getClassLoader()),
                                    new DataWeaveRunnerProvider(localRunner, remoteRunner),
                                    new ModulesAnalyzer(), serializer);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    return id;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ToolingArtifact> getParent() {
    return ofNullable(parentToolingArtifact);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, String> getProperties() {
    return artifact.getProperties();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingService connectivityTestingService() {
    return this.connectivityTestingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataService metadataService() {
    return this.metadataService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataSenseService dataSenseService() {
    return this.dataSenseService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeaveService dataWeaveService() {
    return dataWeaveService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueProviderService valueProviderService() {
    return valueProviderService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void dispose() {
    artifact.dispose();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Feature<ResourceLoader> getResourceLoader() {
    return Feature.enabled(new DefaultResourceLoader(artifact, serializer));
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "id": {
        return serializer.serialize(this.getId());
      }
      case "getParent": {
        return this.getParent();
      }
      case "connectivityTestingService": {
        return this.connectivityTestingService();
      }
      case "metadataService": {
        return this.metadataService();
      }
      case "dataSenseService": {
        return this.dataSenseService();
      }
      case "dataWeaveService": {
        return this.dataWeaveService();
      }
      case "valueProviderService": {
        return this.valueProviderService();
      }
      case "getResourceLoader": {
        return this.getResourceLoader();
      }
      case "dispose": {
        this.dispose();
        return null;
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
