/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.service;

import static java.lang.String.format;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.startIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;

import org.mule.runtime.api.el.DefaultExpressionLanguageFactoryService;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.service.Service;

import java.util.List;

import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.apikit.metadata.api.MetadataService;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of service registry for Tooling.
 */
public class DefaultServiceRegistry implements ServiceRegistry {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultServiceRegistry.class);

  private List<Pair<ArtifactClassLoader, Service>> services;

  private LazyValue<MetadataService> apikitMetadataService;

  public DefaultServiceRegistry(List<Pair<ArtifactClassLoader, Service>> services) {
    this.services = services;
    startServices();

    apikitMetadataService = new LazyValue<>(() -> ((MetadataService) services.stream()
        .filter(pair -> pair.getSecond() instanceof MetadataService)
        .map(pair -> pair.getSecond())
        .findFirst().orElseThrow(() -> new RuntimeException("Apikit metadata service cannot be found"))));

  }

  private void startServices() {
    for (Pair<ArtifactClassLoader, Service> service : services) {
      try {
        startIfNeeded(service.getSecond());
      } catch (MuleException e) {
        LOGGER.warn(format("Failed to start service '%s': %s", service.getSecond().getName(), e.getMessage()),
                    e);
      }
    }
  }

  @Override
  public MetadataService getApikitMetadataService() {
    return apikitMetadataService.get();
  }

  @Override
  public void dispose() {
    services.stream().forEach(service -> {
      try {
        stopIfNeeded(service.getSecond());
      } catch (MuleException e) {
        LOGGER.warn(format("Failed to stop service '%s': %s", service.getSecond().getName(), e.getMessage()), e);
      }
      try {
        service.getFirst().dispose();
      } catch (Exception e) {
        LOGGER.warn(format("Failed to dispose service's class loader '%s': %s", service.getSecond().getName(), e.getMessage()),
                    e);
      }
    });
  }

}
