/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * {@link ExtensionModelService} that use a cache for {@link ExtensionModel}s
 */
public class CachedExtensionModelService implements MuleRuntimeExtensionModelProvider {

  private ExtensionModelServiceCache extensionModelServiceCache;
  private InternalExtensionModelService extensionModelService;

  /**
   * Creates a new instance
   * 
   * @param extensionModelServiceCache the cache for {@link ExtensionModel}s
   * @param extensionModelService the service to use to load {@link ExtensionModel}s in case of a cache miss.
   */
  public CachedExtensionModelService(ExtensionModelServiceCache extensionModelServiceCache,
                                     InternalExtensionModelService extensionModelService) {
    this.extensionModelServiceCache = extensionModelServiceCache;
    this.extensionModelService = extensionModelService;
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return extensionModelServiceCache.loadExtensionInformation(toBundleDescriptor(pluginDescriptor), extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<String> getMinMuleVersion(ArtifactDescriptor pluginDescriptor) {
    return extensionModelServiceCache.loadExtensionInformation(toBundleDescriptor(pluginDescriptor), extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(File plugin) {
    return extensionModelServiceCache.loadExtensionInformation(plugin, extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(BundleDescriptor bundleDescriptor) {
    return extensionModelServiceCache.loadExtensionInformation(bundleDescriptor, extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<String> getMinMuleVersion(File plugin) {
    return extensionModelServiceCache.loadExtensionInformation(plugin, extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<String> getExtensionSchema(File plugin) {
    return extensionModelServiceCache.loadExtensionInformation(plugin, extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public Optional<String> getExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return extensionModelServiceCache.loadExtensionInformation(toBundleDescriptor(pluginDescriptor), extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public List<ExtensionModel> getRuntimeExtensionModels() {
    return extensionModelServiceCache.loadRuntimeExtensionModels(extensionModelService);
  }

  private BundleDescriptor toBundleDescriptor(ArtifactDescriptor pluginDescriptor) {
    return new BundleDescriptor.Builder()
        .setGroupId(pluginDescriptor.getGroupId())
        .setArtifactId(pluginDescriptor.getArtifactId())
        .setVersion(pluginDescriptor.getVersion())
        .setType(pluginDescriptor.getExtension())
        .setClassifier(pluginDescriptor.getClassifier())
        .build();
  }

}
