/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.System.nanoTime;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.concurrent.TimeUnit.NANOSECONDS;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.deployment.model.internal.tooling.ToolingPluginClassLoaderBuilder;
import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.deployment.impl.internal.artifact.ExtensionModelDiscoverer;
import org.mule.runtime.module.deployment.impl.internal.plugin.MuleExtensionModelLoaderManager;
import org.mule.runtime.module.extension.internal.capability.xml.schema.DefaultExtensionSchemaGenerator;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import com.google.common.collect.ImmutableList;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link ExtensionModelService}.
 *
 * @since 4.0
 */
public class DefaultExtensionModelService implements InternalExtensionModelService {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultExtensionModelService.class);

  private ExtensionModelDiscoverer extensionModelDiscoverer = new ExtensionModelDiscoverer();
  private MavenClient mavenClient;
  private MuleArtifactResourcesRegistry muleArtifactResourcesRegistry;
  private ExtensionSchemaGenerator schemaGenerator = new DefaultExtensionSchemaGenerator();

  private List<ExtensionModel> runtimeExtensionModels;

  public DefaultExtensionModelService(MavenClient mavenClient,
                                      MuleArtifactResourcesRegistry muleArtifactResourcesRegistry) {
    requireNonNull(mavenClient, "mavenClient cannot be null");
    requireNonNull(muleArtifactResourcesRegistry, "muleArtifactResourcesRegistry cannot be null");

    this.mavenClient = mavenClient;
    this.muleArtifactResourcesRegistry = muleArtifactResourcesRegistry;
    runtimeExtensionModels = new ArrayList<>(extensionModelDiscoverer.discoverRuntimeExtensionModels());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ExtensionModel> loadRuntimeExtensionModels() {
    return ImmutableList.<ExtensionModel>builder().addAll(runtimeExtensionModels).build();
  }

  private Optional<LoadedExtensionInformation> loadExtensionData(ArtifactPluginDescriptor artifactPluginDescriptor) {
    long startTime = nanoTime();
    ToolingPluginClassLoaderBuilder builder =
        new ToolingPluginClassLoaderBuilder(muleArtifactResourcesRegistry.newTemporaryArtifactClassLoaderFactory(),
                                            muleArtifactResourcesRegistry.getPluginDependenciesResolver(),
                                            artifactPluginDescriptor,
                                            muleArtifactResourcesRegistry.getRegionPluginClassLoadersFactory());
    builder.setParentClassLoader(muleArtifactResourcesRegistry.getContainerArtifactClassLoader());
    ToolingArtifactClassLoader toolingArtifactClassLoader = null;
    try {
      toolingArtifactClassLoader = builder.build();
      MuleExtensionModelLoaderManager extensionModelLoaderRepository =
          new MuleExtensionModelLoaderManager(muleArtifactResourcesRegistry.getContainerArtifactClassLoader());
      extensionModelLoaderRepository.start();
      final Optional<LoadedExtensionInformation> loadedExtensionInformation =
          getLoadedExtensionInformation(artifactPluginDescriptor, toolingArtifactClassLoader, extensionModelLoaderRepository);
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Extension model for {} loaded in {}ms", artifactPluginDescriptor.getBundleDescriptor(),
                     NANOSECONDS.toMillis(nanoTime() - startTime));
      }
      return loadedExtensionInformation;
    } catch (Exception e) {
      throw new ToolingException(e);
    } finally {
      if (toolingArtifactClassLoader != null) {
        toolingArtifactClassLoader.dispose();
      }
    }
  }

  private Optional<LoadedExtensionInformation> getLoadedExtensionInformation(ArtifactPluginDescriptor pluginDescriptor,
                                                                             ToolingArtifactClassLoader toolingArtifactCL,
                                                                             MuleExtensionModelLoaderManager loaderRepository) {
    Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> descriptorsWithExtensions =
        discoverPluginsExtensionModel(toolingArtifactCL, loaderRepository).stream()
            .map(pair -> withContextClassLoader(this.getClass().getClassLoader(), () -> {
              ExtensionModelJsonSerializer extensionModelJsonSerializer = new ExtensionModelJsonSerializer();
              Pair<ArtifactPluginDescriptor, ExtensionModel> result = new Pair<>(pair.getFirst(), extensionModelJsonSerializer
                  .deserialize(extensionModelJsonSerializer.serialize(pair.getSecond())));
              return result;
            }))
            .collect(toSet());
    Optional<ExtensionModel> foundExtension = getExtensionModel(pluginDescriptor, descriptorsWithExtensions);
    if (foundExtension.isPresent()) {
      LazyValue<String> lazySchema = getSchema(descriptorsWithExtensions, foundExtension.get());
      return Optional
          .of(new LoadedExtensionInformation(foundExtension.get(), lazySchema, pluginDescriptor.getMinMuleVersion().toString()));
    } else {
      return empty();
    }
  }

  private LazyValue<String> getSchema(Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> descriptorsWithExtensions,
                                      ExtensionModel foundExtension) {
    Set<ExtensionModel> extensions = descriptorsWithExtensions.stream().map(Pair::getSecond).collect(toSet());
    return new LazyValue<>(() -> schemaGenerator.generate(foundExtension, DslResolvingContext.getDefault(extensions)));
  }

  private Optional<ExtensionModel> getExtensionModel(ArtifactPluginDescriptor descriptor,
                                                     Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> descriptorsWithExtensions) {
    if (descriptorsWithExtensions.isEmpty()) {
      return empty();
    }
    return descriptorsWithExtensions.stream().filter(e -> e.getFirst().equals(descriptor)).map(Pair::getSecond).findFirst();
  }

  private Set<Pair<ArtifactPluginDescriptor, ExtensionModel>> discoverPluginsExtensionModel(ToolingArtifactClassLoader toolingArtifactCL,
                                                                                            MuleExtensionModelLoaderManager extensionModelLoaderRepository) {
    List<Pair<ArtifactPluginDescriptor, ArtifactClassLoader>> artifacts = toolingArtifactCL.getArtifactPluginClassLoaders()
        .stream()
        .map(a -> new Pair<ArtifactPluginDescriptor, ArtifactClassLoader>(a.getArtifactDescriptor(), a))
        .collect(toList());
    return extensionModelDiscoverer.discoverPluginsExtensionModels(extensionModelLoaderRepository, artifacts);
  }

  @Override
  public Optional<LoadedExtensionInformation> loadExtensionData(BundleDescriptor pluginDescriptor) {
    org.mule.maven.client.api.model.BundleDescriptor.Builder builder =
        new org.mule.maven.client.api.model.BundleDescriptor.Builder();
    builder.setGroupId(pluginDescriptor.getGroupId())
        .setArtifactId(pluginDescriptor.getArtifactId())
        .setType(pluginDescriptor.getType())
        .setVersion(pluginDescriptor.getVersion())
        .setClassifier(pluginDescriptor.getClassifier().orElse(null))
        .build();
    final ArtifactPluginDescriptor artifactPluginDescriptor =
        readBundleDescriptor(new File(mavenClient.resolveBundleDescriptor(builder.build())
            .getBundleUri()));
    try {
      return loadExtensionData(artifactPluginDescriptor);
    } catch (ToolingException e) {
      throw e;
    } catch (Exception e) {
      throw new ToolingException(e);
    }
  }

  @Override
  public Optional<LoadedExtensionInformation> loadExtensionData(BundleDescriptor pluginDescriptor, File pluginJarFile) {
    final ArtifactPluginDescriptor artifactPluginDescriptor = readBundleDescriptor(pluginJarFile);
    try {
      return loadExtensionData(artifactPluginDescriptor);
    } catch (ToolingException e) {
      throw e;
    } catch (Exception e) {
      throw new ToolingException(e);
    }
  }

  @Override
  public ArtifactPluginDescriptor readBundleDescriptor(File pluginFile) {
    try {
      ArtifactPluginDescriptor artifactPluginDescriptor;
      artifactPluginDescriptor = muleArtifactResourcesRegistry.getArtifactPluginDescriptorLoader().load(pluginFile);
      return artifactPluginDescriptor;
    } catch (Exception e) {
      throw new ToolingException("Error while loading ExtensionModel for plugin: " + pluginFile.getAbsolutePath(), e);
    }
  }

}
