/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.lang.String.format;
import static org.mule.tooling.client.internal.serialization.TypeAnnotationsTransformers.typeAnnotationTransformers;
import org.mule.metadata.api.annotation.TypeAnnotation;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Writer that transforms the {@link TypeAnnotation} from Tooling Runtime Client to Tooling API.
 *
 * @since 4.2.0
 */
public class TypeAnnotationMapWriter {

  private static Logger LOGGER = LoggerFactory.getLogger(TypeAnnotationMapWriter.class);

  private List<Class> alreadyLoggedNotSupportedTypeAnnotations = new ArrayList<>();
  private ClassLoader targetClassLoader;
  private List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient;

  public TypeAnnotationMapWriter(ClassLoader targetClassLoader, List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient) {
    this.targetClassLoader = targetClassLoader;
    this.typeAnnotationsNotSupportedByToolingRuntimeClient = typeAnnotationsNotSupportedByToolingRuntimeClient;
  }

  public Map write(Map map) {
    if (isTypeAnnotationMap(map)) {
      Map<Class, TypeAnnotation> typeAnnotationMap = (Map<Class, TypeAnnotation>) map;
      Map<Class, TypeAnnotation> toTypeAnnotationTransformedMap = new LinkedHashMap<>();
      for (Map.Entry<Class, TypeAnnotation> entry : typeAnnotationMap.entrySet()) {
        if (entry.getValue().isPublic()) {
          if (!typeAnnotationsNotSupportedByToolingRuntimeClient.contains(entry.getKey())) {
            try {
              if (typeAnnotationTransformers.containsKey(entry.getKey())) {
                TypeAnnotation transformed = typeAnnotationTransformers.get(entry.getKey()).transform(entry.getValue());
                targetClassLoader.loadClass(transformed.getClass().getName());
                toTypeAnnotationTransformedMap.put(transformed.getClass(), transformed);
              } else {
                toTypeAnnotationTransformedMap.put(entry.getKey(), entry.getValue());
              }
            } catch (ClassNotFoundException e) {
              typeAnnotationsNotSupportedByToolingRuntimeClient.add(entry.getKey());
            }
          } else {
            if (!alreadyLoggedNotSupportedTypeAnnotations.contains(entry.getKey())) {
              alreadyLoggedNotSupportedTypeAnnotations.add(entry.getKey());
              LOGGER.warn(format("Ignored TypeAnnotation '%s' as it is not supported by Tooling API", entry.getKey()));
            }
          }
        }
      }
      return toTypeAnnotationTransformedMap;
    }
    return map;
  }

  static boolean isTypeAnnotationMap(Map map) {
    if (map.isEmpty()) {
      return false;
    }
    return map.values().stream().filter(value -> value instanceof TypeAnnotation).findAny().isPresent();
  }

}
