/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.lang.System.nanoTime;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.concurrent.TimeUnit.NANOSECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.tooling.client.internal.utils.ServiceUtils.executeHandlingException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.MuleVersion;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.io.File;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.List;
import java.util.Optional;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cache for {@link ExtensionModel}s
 */
public class ExtensionModelServiceCache implements Disposable {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE =
      "tooling.client.ExtensionModelServiceCache.maximumSize";
  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS =
      "tooling.client.ExtensionModelServiceCache.expireAfterAccess";
  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISKSTORE_PATH =
      "tooling.client.ExtensionModelServiceCache.diskStore.path";

  private static final Logger LOGGER = LoggerFactory.getLogger(ExtensionModelServiceCache.class);
  private static final String DEFAULT_MAX_SIZE = "100";

  private static final String MIN_MULE_VERSION = "minMuleVersion";
  private static final String EXTENSION_MODEL = "extensionModel";

  private MuleVersion muleVersion;
  private File diskStorePath;

  private List<ExtensionModel> runtimeExtensionModels;
  private Cache<BundleDescriptor, Optional<LoadedExtensionInformation>> extensionModelsByArtifact = createExtensionModelCache();

  private Cache<BundleDescriptor, Optional<LoadedExtensionInformation>> createExtensionModelCache() {
    CacheBuilder builder = CacheBuilder.newBuilder().maximumSize(cacheMaximumSize());
    cacheExpireAfterAccess().ifPresent(value -> builder.expireAfterAccess(value, SECONDS));
    return builder.build();
  }

  private Integer cacheMaximumSize() {
    Integer cacheSize = valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE, DEFAULT_MAX_SIZE));
    checkArgument(cacheSize > 0,
                  format("Wrong value %d provided in system property %s, the cache cannot be less that zero",
                         cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE));
    return cacheSize;
  }

  private Optional<Long> cacheExpireAfterAccess() {
    if (getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS) == null) {
      return empty();
    }

    Long cacheSize = Long.valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS));
    checkArgument(cacheSize > 0,
                  format("Wrong value %d provided in system property %s, cacheExpireAfterAccess cannot be less that zero",
                         cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE));
    return of(cacheSize);
  }

  private File diskStorePath() {
    if (getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISKSTORE_PATH) == null) {
      return null;
    }

    File diskStorePath = new File(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISKSTORE_PATH));
    if (!diskStorePath.exists()) {
      boolean created = diskStorePath.mkdirs();
      checkArgument(created,
                    format("Couldn't create folders Extension Model service cache persistent data usinn path",
                           diskStorePath));
    }
    return diskStorePath;
  }

  public ExtensionModelServiceCache(String muleVersion) {
    LOGGER.info("Initialising Extension Model Service cache");
    this.muleVersion = new MuleVersion(muleVersion);
    this.diskStorePath = diskStorePath();
    if (this.diskStorePath != null) {
      this.diskStorePath =
          new File(this.diskStorePath, format("%s.%s", this.muleVersion.getMajor(), this.muleVersion.getMinor()));
      if (!this.diskStorePath.exists()) {
        boolean created = this.diskStorePath.mkdirs();
        checkArgument(created,
                      format("Couldn't create folders Extension Model service cache persistent data using path",
                             this.diskStorePath));
      }
      LOGGER.info("Enabling cache persistence for Extension Model Service on path '{}'", this.diskStorePath.getAbsolutePath());
    } else {
      LOGGER.info("Extension Model Service cache working in memory only");
    }
  }

  /**
   * Loads an extension model from a {@link ArtifactDescriptor}.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param bundleDescriptor the plugin {@link BundleDescriptor}
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the {@link LoadedExtensionInformation}, empty if the plugin does not have an extension model
   */
  public Optional<LoadedExtensionInformation> loadExtensionInformation(BundleDescriptor bundleDescriptor,
                                                                       InternalExtensionModelService extensionModelService) {
    return doLoadExtensionInformation(bundleDescriptor, empty(), extensionModelService);
  }

  private Optional<LoadedExtensionInformation> doLoadExtensionInformation(BundleDescriptor bundleDescriptor,
                                                                          Optional<File> pluginJarFile,
                                                                          InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> extensionModelsByArtifact
        .get(bundleDescriptor, () -> {
          if (diskStorePath != null && existExtensionModelFileFor(bundleDescriptor)) {
            try {
              return loadExtensionModelFromDisk(bundleDescriptor);
            } catch (Exception e) {
              LOGGER.warn(
                          "Error while loading {} Extension Model from file data, error: '{}'. It will be loaded again from jar file and writing to disk again",
                          bundleDescriptor, e.getMessage());
            }
          }
          Optional<LoadedExtensionInformation> extensionInformation = pluginJarFile
              .map(pluginJar -> extensionModelService.loadExtensionData(bundleDescriptor, pluginJar))
              .orElseGet(() -> extensionModelService.loadExtensionData(bundleDescriptor));
          if (diskStorePath != null) {
            writeExtensionModelToDisk(bundleDescriptor, extensionInformation);
          }
          return extensionInformation;
        }));
  }

  private boolean existExtensionModelFileFor(BundleDescriptor bundleDescriptor) {
    return getExtensionModelPersistenceFileFor(bundleDescriptor).exists();
  }

  private Optional<LoadedExtensionInformation> loadExtensionModelFromDisk(BundleDescriptor bundleDescriptor) {
    long startTime = nanoTime();
    try {
      File extensionModelFile = getExtensionModelPersistenceFileFor(bundleDescriptor);
      String content = readFileToString(extensionModelFile, defaultCharset());

      Optional<LoadedExtensionInformation> loadedExtensionInformation = ofNullable(null);

      JSONObject jsonObject = new JSONObject(content);
      if (jsonObject.length() != 0) {
        if (jsonObject.has(MIN_MULE_VERSION) && jsonObject.has(EXTENSION_MODEL)) {
          ExtensionModel extensionModel =
              new ExtensionModelJsonSerializer().deserialize(jsonObject.get(EXTENSION_MODEL).toString());

          File schemaFile = getExtensionSchemaPersistenceFileFor(bundleDescriptor);
          final Reference<String> schemaContent = new Reference<>("");
          if (schemaFile.exists()) {
            if (LOGGER.isInfoEnabled()) {
              LOGGER.info("No schema file found from disk cache for {}", bundleDescriptor);
            }
            schemaContent.set(readFileToString(schemaFile, defaultCharset()));
          }

          loadedExtensionInformation =
              of(new LoadedExtensionInformation(extensionModel, new LazyValue<>(() -> schemaContent.get()),
                                                jsonObject.getString(MIN_MULE_VERSION)));
        } else {
          LOGGER.warn("Ignoring file for {} as its content is invalid", bundleDescriptor);
        }
      }

      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Extension model for {} loaded from file disk in {}ms", bundleDescriptor,
                     NANOSECONDS.toMillis(nanoTime() - startTime));
      }
      return loadedExtensionInformation;
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
  }

  private File getExtensionModelPersistenceFileFor(BundleDescriptor bundleDescriptor) {
    return diskStorePath.toPath().resolve(getPathForBundleDescriptor(bundleDescriptor, "json")).toFile();
  }

  private File getExtensionSchemaPersistenceFileFor(BundleDescriptor bundleDescriptor) {
    return diskStorePath.toPath().resolve(getPathForBundleDescriptor(bundleDescriptor, "xsd")).toFile();
  }

  private void writeExtensionModelToDisk(BundleDescriptor bundleDescriptor,
                                         Optional<LoadedExtensionInformation> extensionInformation) {
    File extensionModelFile = getExtensionModelPersistenceFileFor(bundleDescriptor);
    File extensionSchemaFile = getExtensionSchemaPersistenceFileFor(bundleDescriptor);
    long startTime = nanoTime();
    extensionInformation.ifPresent(extensionModelData -> {
      JSONObject jsonObject = new JSONObject();
      jsonObject.put(MIN_MULE_VERSION, extensionModelData.getMinMuleVersion());
      jsonObject.put(EXTENSION_MODEL,
                     new JSONObject(new ExtensionModelJsonSerializer(false).serialize(extensionModelData.getExtensionModel())));
      try {
        writeStringToFile(extensionModelFile, jsonObject.toString(2), defaultCharset(), false);
        writeStringToFile(extensionSchemaFile, extensionModelData.getSchema().get(), defaultCharset(), false);
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Extension model data for {} written to file disk in {}ms", bundleDescriptor,
                       NANOSECONDS.toMillis(nanoTime() - startTime));
        }
      } catch (IOException e) {
        LOGGER.warn(format("Couldn't write to disk data for Extension Model '{}'", bundleDescriptor), e);
        if (extensionModelFile.exists()) {
          deleteQuietly(extensionModelFile);
        }
        if (extensionSchemaFile.exists()) {
          deleteQuietly(extensionSchemaFile);
        }
      }
    });
  }

  private String getPathForBundleDescriptor(BundleDescriptor bundleDescriptor, String extension) {
    StringBuilder path = new StringBuilder(128);
    path.append(bundleDescriptor.getGroupId().replace('.', '/')).append('/');
    path.append(bundleDescriptor.getArtifactId()).append('/');
    path.append(bundleDescriptor.getVersion()).append('/');
    path.append(bundleDescriptor.getArtifactId()).append('-');
    path.append(bundleDescriptor.getVersion());
    if (bundleDescriptor.getClassifier().isPresent()) {
      path.append('-').append(bundleDescriptor.getClassifier().get());
    }

    path.append('.').append(extension);

    return path.toString();
  }


  /**
   * Loads an extension model from a {@link File} pointing to the plugin artifact.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param plugin the plugin descriptor
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the {@link LoadedExtensionInformation}, empty if the plugin does not have an extension model
   */
  public Optional<LoadedExtensionInformation> loadExtensionInformation(File plugin,
                                                                       InternalExtensionModelService extensionModelService) {
    final ArtifactPluginDescriptor artifactPluginDescriptor = extensionModelService.readBundleDescriptor(plugin);
    return doLoadExtensionInformation(artifactPluginDescriptor.getBundleDescriptor(), of(plugin), extensionModelService);
  }

  /**
   * Loads the core extension model.
   *
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the core extension model
   */
  public List<ExtensionModel> loadRuntimeExtensionModels(InternalExtensionModelService extensionModelService) {
    if (runtimeExtensionModels == null) {
      runtimeExtensionModels = extensionModelService.loadRuntimeExtensionModels();
    }
    return runtimeExtensionModels;
  }

  @Override
  public void dispose() {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Disposing Extension Model Service cache");
    }
    this.extensionModelsByArtifact.invalidateAll();
  }

}
