/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.icons;

import static java.nio.file.FileSystems.getDefault;
import static org.mule.runtime.core.api.util.IOUtils.toByteArray;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.icons.DefaultIconsResult;
import org.mule.tooling.client.api.icons.ExtensionIconsService;
import org.mule.tooling.client.api.icons.IconsResult;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/**
 * Default {@link ExtensionIconsService} implementation
 *
 * @since 1.1.0
 */
public class DefaultExtensionIconsService implements ExtensionIconsService, Command {

  private MavenClient mavenClient;
  private Serializer serializer;
  private static final String SVG = ".svg";

  public DefaultExtensionIconsService(MavenClient mavenClient, Serializer serializer) {
    this.mavenClient = mavenClient;
    this.serializer = serializer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public IconsResult getIconsFor(ArtifactDescriptor artifactDescriptor) {
    BundleDependency bundleDependency = mavenClient.resolveBundleDescriptor(toBundleDescriptor(artifactDescriptor));

    URI bundleUri = bundleDependency.getBundleUri();
    try (ZipFile file = new ZipFile(new File(bundleUri));) {
      Enumeration<? extends ZipEntry> entries = file.entries();

      Map<String, byte[]> additionalIcons = new HashMap<>();
      byte[] defaultIcon = null;
      while (entries.hasMoreElements()) {
        ZipEntry entry = entries.nextElement();
        String iconPath = entry.getName();
        if (iconPath.startsWith("META-INF/mule-artifact") && iconPath.toLowerCase().endsWith(SVG)) {
          int i = iconPath.lastIndexOf(getDefault().getSeparator()) + 1;
          String iconName = iconPath.substring(i, iconPath.length() - 4);
          if (iconName.equals("icon")) {
            defaultIcon = toByteArray(file.getInputStream(entry));
          } else {
            additionalIcons.put(iconName, toByteArray(file.getInputStream(entry)));
          }
        }
      }

      return new DefaultIconsResult(defaultIcon, additionalIcons);
    } catch (IOException e) {
      throw new RuntimeException("Unable to retrieve icons.", e);
    }
  }


  private BundleDescriptor toBundleDescriptor(ArtifactDescriptor pluginDescriptor) {
    return new BundleDescriptor.Builder()
        .setGroupId(pluginDescriptor.getGroupId())
        .setArtifactId(pluginDescriptor.getArtifactId())
        .setVersion(pluginDescriptor.getVersion())
        .setType(pluginDescriptor.getExtension())
        .setClassifier(pluginDescriptor.getClassifier())
        .build();
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    if (methodName.equals("getIconsFor")) {
      Object artifactDescriptor = serializer.deserialize(arguments[0]);
      IconsResult iconsResult = getIconsFor((ArtifactDescriptor) artifactDescriptor);
      return serializer.serialize(iconsResult);
    } else {
      throw methodNotFound(DefaultExtensionIconsService.class, methodName);
    }
  }
}
