/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toCollection;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.tooling.client.internal.CollectionUtils.copy;
import org.mule.runtime.extension.api.property.DefaultImplementingTypeModelProperty;
import org.mule.runtime.extension.api.property.InfrastructureParameterModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyIdModelProperty;
import org.mule.runtime.extension.api.property.MetadataKeyPartModelProperty;
import org.mule.runtime.extension.api.property.QNameModelProperty;
import org.mule.tooling.client.api.extension.model.ClassValueModel;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExpressionSupport;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryType;
import org.mule.tooling.client.api.extension.model.LayoutModel;
import org.mule.tooling.client.api.extension.model.OutputModel;
import org.mule.tooling.client.api.extension.model.ParameterDslConfiguration;
import org.mule.tooling.client.api.extension.model.PathModel;
import org.mule.tooling.client.api.extension.model.PathModelLocation;
import org.mule.tooling.client.api.extension.model.PathModelType;
import org.mule.tooling.client.api.extension.model.StereotypeModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionManagementType;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyIdModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyPartModel;
import org.mule.tooling.client.api.extension.model.nested.NestableElementModel;
import org.mule.tooling.client.api.extension.model.nested.NestedChainModel;
import org.mule.tooling.client.api.extension.model.nested.NestedComponentModel;
import org.mule.tooling.client.api.extension.model.nested.NestedRouteModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.parameter.ExclusiveParametersModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterRole;
import org.mule.tooling.client.api.extension.model.property.DefaultImplementingTypeModel;
import org.mule.tooling.client.api.extension.model.property.InfrastructureParameterModel;
import org.mule.tooling.client.api.extension.model.property.QNameModel;
import org.mule.tooling.client.api.extension.model.source.SourceCallbackModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.api.extension.model.value.ValueProviderModel;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Utils for creating the Extension Model DTOs from the Mule-API objects
 *
 * @since 4.0
 */
public class ExtensionModelPartsFactory {

  public static Set<ExternalLibraryModel> toExternalLibraryModelsDTO(
                                                                     Set<org.mule.runtime.api.meta.model.ExternalLibraryModel> externalLibraryModels) {
    return externalLibraryModels.stream().map(externalLibraryModel -> new ExternalLibraryModel(externalLibraryModel.getName(),
                                                                                               externalLibraryModel
                                                                                                   .getDescription(),
                                                                                               externalLibraryModel
                                                                                                   .getRequiredClassName()
                                                                                                   .orElse(null),
                                                                                               externalLibraryModel
                                                                                                   .getRegexMatcher()
                                                                                                   .orElse(null),
                                                                                               toExternalLibraryTypeDTO(externalLibraryModel
                                                                                                   .getType()),
                                                                                               externalLibraryModel
                                                                                                   .getSuggestedCoordinates()
                                                                                                   .orElse(null),
                                                                                               externalLibraryModel.isOptional()))
        .collect(toSet());
  }

  private static ExternalLibraryType toExternalLibraryTypeDTO(org.mule.runtime.api.meta.ExternalLibraryType type) {
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.NATIVE) {
      return ExternalLibraryType.nativeExternalLibraryType(type.name());
    }
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.JAR) {
      return ExternalLibraryType.jarExternalLibraryType(type.name());
    }
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.DEPENDENCY) {
      return ExternalLibraryType.dependencyExternalLibraryType(type.name());
    }
    return new ExternalLibraryType(type.name());
  }


  public static ConnectionProviderModel toConnectionProviderDTO(org.mule.runtime.api.meta.model.connection.ConnectionProviderModel connectionProviderModel) {
    return new ConnectionProviderModel(connectionProviderModel.getName(),
                                       connectionProviderModel.getDescription(),
                                       toDisplayModelDTO(connectionProviderModel.getDisplayModel()),
                                       toConnectionManagementTypeDTO(connectionProviderModel.getConnectionManagementType()),
                                       connectionProviderModel.supportsConnectivityTesting(),
                                       toParameterGroupModelsDTO(connectionProviderModel.getParameterGroupModels()),
                                       toExternalLibraryModelsDTO(connectionProviderModel.getExternalLibraryModels()),
                                       toStereotypeDTO(connectionProviderModel.getStereotype()));
  }

  private static ConnectionManagementType toConnectionManagementTypeDTO(
                                                                        org.mule.runtime.api.meta.model.connection.ConnectionManagementType connectionManagementType) {
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.POOLING) {
      return ConnectionManagementType.poolingConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.CACHED) {
      return ConnectionManagementType.cachedConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.NONE) {
      return ConnectionManagementType.noneConnectionManagementType(connectionManagementType.name());
    }

    return new ConnectionManagementType(connectionManagementType.name());
  }


  public static SourceModel toSourceModelDTO(org.mule.runtime.api.meta.model.source.SourceModel sourceModel) {
    if (sourceModel == null) {
      return null;
    }
    return new SourceModel(sourceModel.getName(),
                           sourceModel.getDescription(),
                           toParameterGroupModelsDTO(sourceModel.getParameterGroupModels()),
                           toDisplayModelDTO(sourceModel.getDisplayModel()),
                           toOutputModelDTO(sourceModel.getOutput()),
                           toOutputModelDTO(sourceModel.getOutputAttributes()),
                           sourceModel.isTransactional(),
                           sourceModel.requiresConnection(),
                           sourceModel.supportsStreaming(),
                           toStereotypeDTO(sourceModel.getStereotype()),
                           sourceModel.hasResponse(),
                           sourceModel.runsOnPrimaryNodeOnly(),
                           toSourceCallbackModelDTO(sourceModel.getSuccessCallback()),
                           toSourceCallbackModelDTO(sourceModel.getErrorCallback()),
                           toSourceCallbackModelDTO(sourceModel.getTerminateCallback()),
                           toErrorModelsDTO(sourceModel.getErrorModels()),
                           toNestedComponentModelsDTO(sourceModel.getNestedComponents()),
                           toMetadataKeyIdModel(sourceModel.getModelProperty(MetadataKeyIdModelProperty.class)));
  }



  private static List<? extends NestableElementModel> toNestedComponentModelsDTO(
                                                                                 List<? extends org.mule.runtime.api.meta.model.nested.NestableElementModel> components) {
    return components.stream().map(model -> toNestedComponentModelDTO(model)).collect(toList());
  }

  private static NestableElementModel toNestedComponentModelDTO(org.mule.runtime.api.meta.model.nested.NestableElementModel nestedModel) {
    final NestableElementModelDTOVisitor nestableElementModelVisitor = new NestableElementModelDTOVisitor();
    nestedModel.accept(nestableElementModelVisitor);
    return nestableElementModelVisitor.getNestableElementModelDTO();
  }

  private static class NestableElementModelDTOVisitor
      implements org.mule.runtime.api.meta.model.nested.NestableElementModelVisitor {

    private NestableElementModel nestableElementModelDTO = null;

    public NestableElementModel getNestableElementModelDTO() {
      return nestableElementModelDTO;
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.nested.NestedComponentModel nestedComponentModel) {
      nestableElementModelDTO = new NestedComponentModel(nestedComponentModel.getName(),
                                                         nestedComponentModel.getDescription(),
                                                         toDisplayModelDTO(nestedComponentModel.getDisplayModel()),
                                                         nestedComponentModel.isRequired(),
                                                         toStereotypesDTO(nestedComponentModel.getAllowedStereotypes()));
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.nested.NestedChainModel nestedChainModel) {
      nestableElementModelDTO = new NestedChainModel(nestedChainModel.getName(),
                                                     nestedChainModel.getDescription(),
                                                     toDisplayModelDTO(nestedChainModel.getDisplayModel()),
                                                     nestedChainModel.isRequired(),
                                                     toStereotypesDTO(nestedChainModel.getAllowedStereotypes()));
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.nested.NestedRouteModel nestedRouteModel) {
      nestableElementModelDTO = new NestedRouteModel(nestedRouteModel.getName(),
                                                     nestedRouteModel.getDescription(),
                                                     toDisplayModelDTO(nestedRouteModel.getDisplayModel()),
                                                     nestedRouteModel.getMinOccurs(),
                                                     nestedRouteModel.getMaxOccurs()
                                                         .orElse(null),
                                                     toParameterGroupModelsDTO(nestedRouteModel.getParameterGroupModels()),
                                                     toNestedComponentModelsDTO(nestedRouteModel.getNestedComponents()));
    }
  }

  private static Set<ErrorModel> toErrorModelsDTO(Set<org.mule.runtime.api.meta.model.error.ErrorModel> errorModels) {
    return errorModels.stream()
        .map(errorModel -> toErrorModelDTO(errorModel))
        .collect(toSet());
  }

  private static ErrorModel toErrorModelDTO(org.mule.runtime.api.meta.model.error.ErrorModel errorModel) {
    if (errorModel == null) {
      return null;
    }
    return new ErrorModel(errorModel.getType(), errorModel.getNamespace(), errorModel.isHandleable(),
                          toErrorModelDTO(errorModel.getParent().orElse(null)));
  }

  private static SourceCallbackModel toSourceCallbackModelDTO(Optional<org.mule.runtime.api.meta.model.source.SourceCallbackModel> sourceCallbackModelOptional) {
    return sourceCallbackModelOptional
        .map(sourceCallbackModel -> new SourceCallbackModel(sourceCallbackModel.getName(),
                                                            sourceCallbackModel.getDescription(),
                                                            toParameterGroupModelsDTO(sourceCallbackModel
                                                                .getParameterGroupModels()),
                                                            toDisplayModelDTO(sourceCallbackModel.getDisplayModel())))
        .orElse(null);
  }

  private static Set<StereotypeModel> toStereotypesDTO(Collection<org.mule.runtime.api.meta.model.stereotype.StereotypeModel> stereotypes) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotypes == null) {
      return emptySet();
    }
    return stereotypes.stream()
        .map(stereotype -> new StereotypeModel(stereotype.getType(), stereotype.getNamespace(),
                                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO)
                                                   .orElse(null)))
        .collect(toCollection(LinkedHashSet::new));
  }

  public static <T extends Collection<StereotypeModel>> T toStereotypesDTO(Collection<org.mule.runtime.api.meta.model.stereotype.StereotypeModel> stereotypes,
                                                                           T output) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotypes == null) {
      return output;
    }
    return stereotypes.stream()
        .map(stereotype -> new StereotypeModel(stereotype.getType(), stereotype.getNamespace(),
                                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO)
                                                   .orElse(null)))
        .collect(toCollection(() -> output));
  }

  static StereotypeModel toStereotypeDTO(org.mule.runtime.api.meta.model.stereotype.StereotypeModel stereotype) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotype == null) {
      return null;
    }
    return new StereotypeModel(stereotype.getType(), stereotype.getNamespace(),
                               stereotype.getParent().map(ExtensionModelPartsFactory::toStereotypeDTO).orElse(null));
  }

  private static OutputModel toOutputModelDTO(org.mule.runtime.api.meta.model.OutputModel output) {
    return new OutputModel(output.getDescription(), output.getType(), output.hasDynamicType());
  }

  public static OperationModel toOperationModelDTO(org.mule.runtime.api.meta.model.operation.OperationModel operationModel) {
    if (operationModel == null) {
      return null;
    }

    return new OperationModel(operationModel.getName(),
                              operationModel.getDescription(),
                              toParameterGroupModelsDTO(operationModel.getParameterGroupModels()),
                              toDisplayModelDTO(operationModel.getDisplayModel()),
                              operationModel.isBlocking(),
                              toErrorModelsDTO(operationModel.getErrorModels()),
                              toOutputModelDTO(operationModel.getOutput()),
                              toOutputModelDTO(operationModel.getOutputAttributes()),
                              operationModel.isTransactional(),
                              operationModel.requiresConnection(),
                              operationModel.supportsStreaming(),
                              toStereotypeDTO(operationModel.getStereotype()),
                              toNestedComponentModelsDTO(operationModel.getNestedComponents()),
                              toMetadataKeyIdModel(operationModel.getModelProperty(MetadataKeyIdModelProperty.class)));
  }

  public static List<ParameterGroupModel> toParameterGroupModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels) {
    return parameterGroupModels.stream()
        .map(parameterGroupModel -> new ParameterGroupModel(parameterGroupModel.getName(),
                                                            parameterGroupModel.getDescription(),
                                                            toDisplayModelDTO(parameterGroupModel.getDisplayModel()),
                                                            toLayoutModelDTO(parameterGroupModel.getLayoutModel()),
                                                            toParameterModelsDTO(parameterGroupModel.getParameterModels()),
                                                            toExclusiveParameterModelsDTO(parameterGroupModel
                                                                .getExclusiveParametersModels()),
                                                            parameterGroupModel.isShowInDsl()))
        .collect(toList());
  }

  private static List<ExclusiveParametersModel> toExclusiveParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ExclusiveParametersModel> exclusiveParametersModels) {
    return exclusiveParametersModels.stream()
        .map(exclusiveParametersModel -> new ExclusiveParametersModel(copy(exclusiveParametersModel
            .getExclusiveParameterNames()),
                                                                      exclusiveParametersModel.isOneRequired()))
        .collect(toList());
  }

  private static List<ParameterModel> toParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels) {
    return parameterModels.stream()
        .map(parameterModel -> new ParameterModel(parameterModel.getName(),
                                                  parameterModel.getDescription(),
                                                  toDisplayModelDTO(parameterModel.getDisplayModel()),
                                                  parameterModel.getType(),
                                                  parameterModel.hasDynamicType(),
                                                  parameterModel.isRequired(),
                                                  parameterModel.isOverrideFromConfig(),
                                                  toExpressionSupportDTO(parameterModel.getExpressionSupport()),
                                                  parameterModel.getDefaultValue(),
                                                  toParameterRoleDTO(parameterModel.getRole()),
                                                  toParameterDslConfigurationDTO(parameterModel.getDslConfiguration()),
                                                  toLayoutModelDTO(parameterModel.getLayoutModel()),
                                                  toMetadataKeyPartModel(parameterModel
                                                      .getModelProperty(MetadataKeyPartModelProperty.class)),
                                                  toQNameModel(parameterModel.getModelProperty(QNameModelProperty.class)),
                                                  toInfrastructureParameterModel(parameterModel
                                                      .getModelProperty(InfrastructureParameterModelProperty.class)),
                                                  toDefaultImplementingTypeModel(parameterModel
                                                      .getModelProperty(DefaultImplementingTypeModelProperty.class)),
                                                  toValueProviderModelDTO(parameterModel.getValueProviderModel()),
                                                  toStereotypesDTO(parameterModel.getAllowedStereotypes(), new ArrayList<>())))
        .collect(toList());
  }

  private static QNameModel toQNameModel(Optional<QNameModelProperty> modelProperty) {
    return modelProperty.map(qNameProperty -> new QNameModel(qNameProperty.getValue()))
        .orElse(null);
  }

  private static DefaultImplementingTypeModel toDefaultImplementingTypeModel(Optional<DefaultImplementingTypeModelProperty> modelProperty) {
    return modelProperty
        .map(defaultImplementingTypeModelProperty -> new DefaultImplementingTypeModel(defaultImplementingTypeModelProperty
            .value()))
        .orElse(null);
  }

  private static InfrastructureParameterModel toInfrastructureParameterModel(Optional<InfrastructureParameterModelProperty> modelProperty) {
    return modelProperty
        .map(infrastructureParameterModelProperty -> new InfrastructureParameterModel(infrastructureParameterModelProperty
            .getSequence()))
        .orElse(null);
  }

  private static MetadataKeyPartModel toMetadataKeyPartModel(Optional<MetadataKeyPartModelProperty> modelProperty) {
    return modelProperty.map(metadataKeyPartModelProperty -> new MetadataKeyPartModel(metadataKeyPartModelProperty.getOrder(),
                                                                                      metadataKeyPartModelProperty
                                                                                          .isProvidedByKeyResolver()))
        .orElse(null);
  }

  private static MetadataKeyIdModel toMetadataKeyIdModel(Optional<MetadataKeyIdModelProperty> modelProperty) {
    return modelProperty
        .map(metadataKeyIdModel -> new MetadataKeyIdModel(metadataKeyIdModel.getType(), metadataKeyIdModel.getParameterName(),
                                                          metadataKeyIdModel.getCategoryName().orElse(null)))
        .orElse(null);
  }

  public static ExpressionSupport toExpressionSupportDTO(org.mule.runtime.api.meta.ExpressionSupport expressionSupport) {
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED) {
      return ExpressionSupport.supportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED) {
      return ExpressionSupport.notSupportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.REQUIRED) {
      return ExpressionSupport.requiredExpressionSupport(expressionSupport.name());
    }
    return new ExpressionSupport(expressionSupport.name());
  }

  private static ParameterDslConfiguration toParameterDslConfigurationDTO(
                                                                          org.mule.runtime.api.meta.model.ParameterDslConfiguration dslConfiguration) {
    return new ParameterDslConfiguration(dslConfiguration.allowsInlineDefinition(), dslConfiguration.allowTopLevelDefinition(),
                                         dslConfiguration.allowsReferences());
  }

  public static LayoutModel toLayoutModelDTO(Optional<org.mule.runtime.api.meta.model.display.LayoutModel> layoutModelOptional) {
    return layoutModelOptional.map(layoutModel -> new LayoutModel(layoutModel.getTabName().orElse(null),
                                                                  layoutModel.getOrder().orElse(null),
                                                                  layoutModel.isPassword(),
                                                                  layoutModel.isText(),
                                                                  layoutModel.isQuery()))
        .orElse(null);
  }

  private static ParameterRole toParameterRoleDTO(org.mule.runtime.api.meta.model.parameter.ParameterRole parameterRole) {
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR) {
      return ParameterRole.behaviourParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT) {
      return ParameterRole.contentParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.PRIMARY_CONTENT) {
      return ParameterRole.primaryContentParameterRole(parameterRole.name());
    }
    return new ParameterRole(parameterRole.name());
  }

  public static DisplayModel toDisplayModelDTO(Optional<org.mule.runtime.api.meta.model.display.DisplayModel> displayModelOptional) {
    return displayModelOptional.map(displayModel -> new DisplayModel(displayModel.getDisplayName(), displayModel.getSummary(),
                                                                     displayModel.getExample(),
                                                                     toPathModelDTO(displayModel.getPathModel()),
                                                                     toClassValueModelDTO(displayModel.getClassValueModel())))
        .orElse(null);
  }

  public static ClassValueModel toClassValueModelDTO(Optional<org.mule.runtime.api.meta.model.display.ClassValueModel> classValue) {
    return classValue.map(cv -> new ClassValueModel(cv.getAssignableFrom())).orElse(null);
  }

  public static PathModel toPathModelDTO(Optional<org.mule.runtime.api.meta.model.display.PathModel> pathModelOptional) {
    return pathModelOptional.map(pathModel -> new PathModel(toPathModelTypeDTO(pathModel.getType()), pathModel.acceptsUrls(),
                                                            toPathModelLocationDTO(pathModel.getLocation()),
                                                            pathModel.getFileExtensions().toArray(new String[0])))
        .orElse(null);
  }

  private static PathModelType toPathModelTypeDTO(org.mule.runtime.api.meta.model.display.PathModel.Type type) {
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.DIRECTORY) {
      return PathModelType.directoryPathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.FILE) {
      return PathModelType.filePathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.ANY) {
      return PathModelType.anyPathModelType(type.name());
    }
    return new PathModelType(type.name());
  }

  private static PathModelLocation toPathModelLocationDTO(org.mule.runtime.api.meta.model.display.PathModel.Location location) {
    if (location == org.mule.runtime.api.meta.model.display.PathModel.Location.EMBEDDED) {
      return PathModelLocation.embeddedPathModelLocation(location.name());
    }
    if (location == org.mule.runtime.api.meta.model.display.PathModel.Location.EXTERNAL) {
      return PathModelLocation.externalPathModelLocation(location.name());
    }
    if (location == org.mule.runtime.api.meta.model.display.PathModel.Location.ANY) {
      return PathModelLocation.anyPathModelLocation(location.name());
    }
    return new PathModelLocation(location.name());
  }

  private static ValueProviderModel toValueProviderModelDTO(Optional<org.mule.runtime.api.meta.model.parameter.ValueProviderModel> valueProviderModel) {
    return valueProviderModel
        .map(model -> new ValueProviderModel(model.getActingParameters(), model.requiresConfiguration(),
                                             model.requiresConnection(), model.isOpen(), model.getPartOrder(),
                                             model.getProviderName()))
        .orElse(null);
  }

  public static FunctionModel toFunctionModelDTO(org.mule.runtime.api.meta.model.function.FunctionModel functionModel) {
    return new FunctionModel(functionModel.getName(), functionModel.getDescription(),
                             toParameterGroupModelsDTO(functionModel.getParameterGroupModels()),
                             toDisplayModelDTO(functionModel.getDisplayModel()), toOutputModelDTO(functionModel.getOutput()));
  }


  public static ConstructModel toConstructModelDTO(org.mule.runtime.api.meta.model.construct.ConstructModel constructModel) {
    return new ConstructModel(constructModel.getName(), constructModel.getDescription(),
                              toParameterGroupModelsDTO(constructModel.getParameterGroupModels()),
                              toDisplayModelDTO(constructModel.getDisplayModel()),
                              toStereotypeDTO(constructModel.getStereotype()),
                              toNestedComponentModelsDTO(constructModel.getNestedComponents()),
                              constructModel.allowsTopLevelDeclaration(),
                              toErrorModelsDTO(constructModel.getErrorModels()));
  }

}
