/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Collections.synchronizedList;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.tooling.client.api.exception.ToolingException;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.MapSerializer;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * Serialization support that allows to serialize/deserialze objects from this implementation using Kryo.
 *
 * @since 4.2
 */
public class KryoServerSerializer implements Serializer {

  public static String NAME = "kryo";

  private ClassLoader sourceClassLoader;
  private ClassLoader targetClassLoader;
  private List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient;

  public KryoServerSerializer(ClassLoader sourceClassLoader, ClassLoader targetClassLoader) {
    this.sourceClassLoader = sourceClassLoader;
    this.targetClassLoader = targetClassLoader;
    this.typeAnnotationsNotSupportedByToolingRuntimeClient = synchronizedList(new ArrayList<>());
  }

  private static Kryo createKryo(ClassLoader sourceClassLoader, ClassLoader targetClassLoader,
                                 List<Class> typeAnnotationsNotsupportedByToolingRuntimeClient) {
    Kryo kryo = KryoFactory.createKryo(targetClassLoader);
    kryo.setClassLoader(sourceClassLoader);
    kryo.addDefaultSerializer(DefaultArrayType.class, new DefaultArrayTypeSerializer());
    // org.mule.metadata.api.builder.AbstractBuilder.annotations uses a LinkedHashMap
    kryo.register(LinkedHashMap.class,
                  new TypeAnnotationsMapSerializer(targetClassLoader, typeAnnotationsNotsupportedByToolingRuntimeClient));
    return kryo;
  }

  @Override
  public String serialize(Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        createKryo(sourceClassLoader, targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient)
            .writeClassAndObject(output, object);
      }
      return Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  public String safeSerialize(Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        createKryo(sourceClassLoader, targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient).writeObject(output,
                                                                                                                        object);
      }
      return Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  @Override
  public <T> T deserialize(String content) {
    try (Input input = new Input(new ByteArrayInputStream(Base64.getDecoder().decode(content)))) {
      return (T) createKryo(sourceClassLoader, targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient)
          .readClassAndObject(input);
    }
  }

  public <T> T deserialize(String content, Class<T> clazz) {
    try (Input input = new Input(new ByteArrayInputStream(Base64.getDecoder().decode(content)))) {
      return (T) createKryo(sourceClassLoader, targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient)
          .readObject(input, clazz);
    }
  }

  private static class TypeAnnotationsMapSerializer extends MapSerializer {

    private TypeAnnotationMapWriter typeAnnotationMapWriter;
    private TypeAnnotationMapReader typeAnnotationMapReader;

    public TypeAnnotationsMapSerializer(ClassLoader targetClassLoader,
                                        List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient) {
      this.typeAnnotationMapWriter =
          new TypeAnnotationMapWriter(targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient);
      this.typeAnnotationMapReader = new TypeAnnotationMapReader();
    }

    @Override
    public void write(Kryo kryo, Output output, Map map) {
      super.write(kryo, output, typeAnnotationMapWriter.write(map));
    }

    @Override
    public Map read(Kryo kryo, Input input, Class<Map> type) {
      Map original = super.read(kryo, input, type);
      return typeAnnotationMapReader.read(original);
    }

  }

}
