/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.deployment.model.internal.plugin.PluginDependenciesResolver;
import org.mule.runtime.module.deployment.impl.internal.application.ToolingApplicationDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.domain.DomainDescriptorFactory;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.internal.application.ApplicationClassLoaderFactory;
import org.mule.tooling.client.internal.application.DomainClassLoaderFactory;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.service.ServiceRegistry;

import java.util.Optional;

/**
 * Default implementation of {@link ToolingArtifactContext}.
 */
public class DefaultToolingArtifactContext implements ToolingArtifactContext {

  private MuleArtifactResourcesRegistry muleArtifactResourcesRegistry;
  private SpiRuntimeToolingServiceProvider runtimeToolingServiceProvider = new SpiRuntimeToolingServiceProvider();
  private transient volatile boolean runtimeToolingServiceInitialized;
  private RuntimeToolingService runtimeToolingService;

  private Optional<AgentConfiguration> agentConfigurationOptional;
  private ApplicationCache applicationCache;
  private DomainCache domainCache;
  private MavenClient mavenClient;
  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private Optional<MetadataCacheFactory> metadataCacheFactory;
  private Serializer serializer;
  private ServiceRegistry serviceRegistry;

  public DefaultToolingArtifactContext(MuleArtifactResourcesRegistry muleArtifactResourcesRegistry) {
    this.muleArtifactResourcesRegistry = muleArtifactResourcesRegistry;
  }

  public void setAgentConfiguration(Optional<AgentConfiguration> agentConfigurationOptional) {
    this.agentConfigurationOptional = agentConfigurationOptional;
  }

  public Optional<AgentConfiguration> getAgentConfiguration() {
    return agentConfigurationOptional;
  }

  public RuntimeToolingService getRuntimeToolingService() {
    if (!this.runtimeToolingServiceInitialized) {
      synchronized (this) {
        if (!this.runtimeToolingServiceInitialized) {
          this.runtimeToolingService = runtimeToolingServiceProvider.lookUpRuntimeToolingService();

          if (!agentConfigurationOptional.isPresent()) {
            throw new MissingToolingConfigurationException("Tooling Client has to be configured with a configuration to access REST Tooling API in order to resolve operations");
          }

          AgentConfiguration agentConfiguration = agentConfigurationOptional.get();
          runtimeToolingService.setToolingApiUrl(agentConfiguration.getToolingApiUrl(),
                                                 agentConfiguration.getDefaultConnectTimeout(),
                                                 agentConfiguration.getDefaultReadTimeout(),
                                                 agentConfiguration.getSslConfiguration(),
                                                 agentConfiguration.getProxyConfig());

          // Defines the API version of the Agent
          runtimeToolingService.setMuleVersion(agentConfiguration.muleVersion().orElse(null));

          this.runtimeToolingServiceInitialized = true;
        }
      }
    }
    return runtimeToolingService;
  }

  @Override
  public PluginDependenciesResolver getPluginDependenciesResolver() {
    return muleArtifactResourcesRegistry.getPluginDependenciesResolver();
  }

  public ApplicationCache getApplicationCache() {
    return applicationCache;
  }

  public void setApplicationCache(ApplicationCache applicationCache) {
    this.applicationCache = applicationCache;
  }

  public DomainCache getDomainCache() {
    return domainCache;
  }

  public void setDomainCache(DomainCache domainCache) {
    this.domainCache = domainCache;
  }

  public void setMavenClient(MavenClient mavenClient) {
    this.mavenClient = mavenClient;
  }

  public MavenClient getMavenClient() {
    return mavenClient;
  }

  public void setMuleRuntimeExtensionModelProvider(MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider) {
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
  }

  public MuleRuntimeExtensionModelProvider getMuleRuntimeExtensionModelProvider() {
    return muleRuntimeExtensionModelProvider;
  }

  public DomainClassLoaderFactory getDomainClassLoaderFactory() {
    return muleArtifactResourcesRegistry.getDomainClassLoaderFactory();
  }

  public ApplicationClassLoaderFactory getApplicationClassLoaderFactory() {
    return muleArtifactResourcesRegistry.getApplicationClassLoaderFactory();
  }

  public DomainDescriptorFactory getDomainDescriptorFactory() {
    return muleArtifactResourcesRegistry.getDomainDescriptorFactory();
  }

  public ToolingApplicationDescriptorFactory getApplicationDescriptorFactory() {
    return muleArtifactResourcesRegistry.getApplicationDescriptorFactory();
  }

  public void setMetadataCacheFactory(Optional<MetadataCacheFactory> metadataCacheFactory) {
    this.metadataCacheFactory = metadataCacheFactory;
  }

  public Optional<MetadataCacheFactory> getMetadataCacheFactory() {
    return metadataCacheFactory;
  }

  public void setSerializer(Serializer serializer) {
    this.serializer = serializer;
  }

  public Serializer getSerializer() {
    return serializer;
  }

  public void setServiceRegistry(ServiceRegistry serviceRegistry) {
    this.serviceRegistry = serviceRegistry;
  }

  public ServiceRegistry getServiceRegistry() {
    return serviceRegistry;
  }
}
