/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.internal.application.RemoteDomainInvoker;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Internal implementation of {@link MetadataProvider}, that is used with DataSense and uses Mule-API objects directly.
 *
 * @since 4.0
 */
public class InternalDomainMetadataProvider implements MetadataProvider {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private RemoteDomainInvoker remoteDomainInvoker;

  /**
   * Creates an instance of the provider.
   *
   * @param remoteDomainInvoker {@link RemoteDomainInvoker} to deploy and resolve operations against a Mule domain. Non null.
   */
  public InternalDomainMetadataProvider(RemoteDomainInvoker remoteDomainInvoker) {
    checkNotNull(remoteDomainInvoker, "remoteDomainInvoker cannot be null");

    this.remoteDomainInvoker = remoteDomainInvoker;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException, ToolingException {
    if (logger.isDebugEnabled()) {
      logger.debug("Getting metadata keys for: {} on domain: {}", metadataKeysRequest, remoteDomainInvoker);
    }

    try {
      return remoteDomainInvoker.evaluateWithRemoteDomain((domainId,
                                                           runtimeToolingService) -> runtimeToolingService
                                                               .getMetadataKeysDomain(domainId,
                                                                                      metadataKeysRequest.getLocation()
                                                                                          .toString(),
                                                                                      metadataKeysRequest
                                                                                          .getRequestTimeout()));
    } catch (Exception e) {
      return failure(newFailure(e).onKeys());
    }
  }

  @Override
  public void disposeMetadataCache(String hashKey) {
    if (logger.isDebugEnabled()) {
      logger.debug("Dispose metadata cache for hashKey: {} on domain: {}", hashKey, remoteDomainInvoker);
    }

    remoteDomainInvoker.evaluateWithRemoteDomain((applicationId, runtimeToolingService) -> {
      runtimeToolingService.disposeDomainMetadataCache(applicationId, hashKey);
      return null;
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(MetadataRequest metadataRequest) {
    throw new UnsupportedOperationException("Metadata for operations cannot be retrieved on domain artifact");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(MetadataRequest metadataRequest) {
    throw new UnsupportedOperationException("Metadata for sources cannot be retrieved on domain artifact");
  }

}
