/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.api.metadata.resolving.FailureCode.COMPONENT_NOT_FOUND;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataFailuresDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataKeysContainerDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataResultDTO;
import static org.mule.tooling.client.internal.serialization.KryoFactory.createKryo;
import org.mule.datasense.enrichment.model.IdComponentModelSelector;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.config.api.dsl.model.metadata.ModelBasedMetadataCacheIdGeneratorFactory;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGenerator;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGeneratorFactory;
import org.mule.runtime.dsl.api.component.config.ComponentConfiguration;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.ImmutableMetadataCacheKeyInfo;
import org.mule.tooling.client.api.datasense.ImmutableMetadataResult;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.metadata.MetadataFailure;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.internal.action.DispacheableAction;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Public implementation of {@link MetadataService} that adapts Mule-API objects to DTOs
 *
 * @since 4.0
 */
public class ToolingMetadataServiceAdapter implements MetadataService, Command {

  private LazyValue<ApplicationModel> applicationModel;
  private LazyValue<MetadataCacheIdGeneratorFactory.ComponentLocator<ComponentConfiguration>> componentLocator;
  private LazyValue<MetadataCacheIdGenerator<ComponentConfiguration>> metadataCacheIdGenerator;

  private MetadataProvider metadataProvider;
  private LazyValue<MetadataCache> metadataCache;
  private Map<String, String> toolingArtifactProperties;

  private Kryo kryo;
  private Serializer serializer;

  /**
   * Creates an instance of this service adapter.
   *
   * @param applicationModel {@link ApplicationModel} to resolve the {@link ComponentModel}. Non null.
   * @param metadataProvider {@link MetadataProvider} for resolving metadata using Mule Runtime. Non null.
   * @param metadataCache {@link MetadataCache} to fetch Metadata from cache. Non null.
   */
  public ToolingMetadataServiceAdapter(LazyValue<ApplicationModel> applicationModel,
                                       LazyValue<Set<ExtensionModel>> extensionModels,
                                       MetadataProvider metadataProvider, LazyValue<MetadataCache> metadataCache,
                                       Map<String, String> toolingArtifactProperties,
                                       Serializer serializer) {
    checkNotNull(applicationModel, "applicationModel cannot be null");
    checkNotNull(metadataProvider, "metadataProvider cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");
    checkNotNull(toolingArtifactProperties, "toolingArtifactProperties, cannot be null");

    this.applicationModel = applicationModel;
    this.metadataProvider = metadataProvider;
    this.metadataCache = metadataCache;
    this.toolingArtifactProperties = toolingArtifactProperties;

    this.serializer = serializer;
    this.kryo = createKryo();

    this.componentLocator = new LazyValue<>(() -> {
      Map<org.mule.runtime.api.component.location.Location, ComponentModel> components = new HashMap<>();
      applicationModel.get().executeOnEveryComponentTree(componentModel -> {
        if (componentModel.getComponentLocation() != null) {
          components.put(org.mule.runtime.api.component.location.Location.builderFromStringRepresentation(componentModel
              .getComponentLocation()
              .getLocation())
              .build(),
                         componentModel);
        }
      });
      return (location -> ofNullable(components.get(location)).map(componentModel -> componentModel.getConfiguration()));
    });

    this.metadataCacheIdGenerator =
        new LazyValue<>(() -> new ModelBasedMetadataCacheIdGeneratorFactory()
            .create(DslResolvingContext.getDefault(extensionModels.get()), componentLocator.get()));
  }

  public String serialize(Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        kryo.writeClassAndObject(output, object);
      }
      return Base64.getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  public <T> T deserialize(String content) {
    try (Input input = new Input(new ByteArrayInputStream(Base64.getDecoder().decode(content)))) {
      return (T) kryo.readClassAndObject(input);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public org.mule.tooling.client.api.metadata.MetadataResult<org.mule.tooling.client.api.metadata.MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException {

    Optional<ComponentModel> componentModel = findComponentModel(applicationModel.get(), metadataKeysRequest.getLocation());

    MetadataResult<MetadataKeysContainer> metadataKeys;
    if (!componentModel.isPresent()) {
      metadataKeys = failure(newFailure()
          .withMessage(format("No object found at location %s", metadataKeysRequest.getLocation().toString())).withFailureCode(
                                                                                                                               COMPONENT_NOT_FOUND)
          .onKeys());
    } else {
      String componentId = IdComponentModelSelector.getComponentId(componentModel.get());
      metadataKeys = deserialize((String) metadataCache.get()
          .getMetadataKeys(new ImmutableMetadataCacheKeyInfo(componentId,
                                                             metadataKeysRequest
                                                                 .getLocation()
                                                                 .toString(),
                                                             null,
                                                             toolingArtifactProperties),
                           () -> {
                             final MetadataResult<MetadataKeysContainer> runtimeMetadataResult =
                                 metadataProvider
                                     .getMetadataKeys(metadataKeysRequest);
                             List<MetadataFailure> failures = emptyList();
                             if (!runtimeMetadataResult.isSuccess()) {
                               failures = toMetadataFailuresDTO(runtimeMetadataResult.getFailures());
                             }
                             return new ImmutableMetadataResult(runtimeMetadataResult
                                 .isSuccess(), serialize(runtimeMetadataResult), failures);
                           }));
    }

    return toMetadataResultDTO(metadataKeys, () -> toMetadataKeysContainerDTO(metadataKeys.get()));
  }

  private Optional<ComponentModel> findComponentModel(ApplicationModel applicationModel, Location location) {
    final Reference<ComponentModel> foundComponentModelReference = new Reference<>();
    Optional<ComponentModel> globalComponent = applicationModel.findTopLevelNamedComponent(location.getGlobalName());
    globalComponent.ifPresent(componentModel -> findComponentWithLocation(componentModel, location)
        .ifPresent(foundComponentModel -> foundComponentModelReference.set(foundComponentModel)));
    return ofNullable(foundComponentModelReference.get());
  }

  private Optional<ComponentModel> findComponentWithLocation(ComponentModel componentModel, Location location) {
    if (componentModel.getComponentLocation().getLocation().equals(location.toString())) {
      return of(componentModel);
    }
    for (ComponentModel childComponent : componentModel.getInnerComponents()) {
      Optional<ComponentModel> foundComponent = findComponentWithLocation(childComponent, location);
      if (foundComponent.isPresent()) {
        return foundComponent;
      }
    }
    return empty();
  }

  @Override
  public Feature<Action<Location>> disposeMetadataCache() {
    return Feature.enabled(new DispacheableAction<>(location -> ToolingMetadataServiceAdapter.this.componentLocator.get()
        .get(org.mule.runtime.api.component.location.Location.builderFromStringRepresentation(location.toString()).build())
        .ifPresent(componentConfiguration -> metadataCacheIdGenerator.get().getIdForGlobalMetadata(componentConfiguration)
            .ifPresent(metadataCacheId -> metadataProvider.disposeMetadataCache(metadataCacheId.getValue()))), Location.class,
                                                    serializer));
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getMetadataKeys": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(MetadataKeysRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(getMetadataKeys(serializer.deserialize(arguments[0])));
      }
      case "disposeMetadataCache": {
        return disposeMetadataCache();
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }
}
