/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.agent.RuntimeToolingService;

import java.io.File;
import java.io.InputStream;
import java.util.Map;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base class implementation for {@link ArtifactDeployer}.
 *
 * @since 4.0
 */
public abstract class AbstractArtifactDeployer implements ArtifactDeployer {

  protected final Logger LOGGER = LoggerFactory.getLogger(this.getClass());

  private ArtifactResources artifactResources;
  private Supplier<RuntimeToolingService> runtimeToolingService;
  private Map<String, String> deploymentProperties;

  /**
   * Creates an instance of the context.
   *
   * @param artifactResources {@link ArtifactResources} of the artifact to be deployed. Non null.
   * @param runtimeToolingService {@link RuntimeToolingService} for resolving operations against the Mule Runtime. Non null.
   * @param deploymentProperties {@link Map} with the deployment properties for the artifact. Non null.
   */
  public AbstractArtifactDeployer(ArtifactResources artifactResources,
                                  Supplier<RuntimeToolingService> runtimeToolingService,
                                  Map<String, String> deploymentProperties) {
    requireNonNull(artifactResources, "artifactResources cannot be null");
    requireNonNull(runtimeToolingService, "runtimeToolingService cannot be null");
    requireNonNull(deploymentProperties, "deploymentProperties cannot be null");

    this.artifactResources = artifactResources;
    this.runtimeToolingService = runtimeToolingService;
    this.deploymentProperties = deploymentProperties;
  }

  @Override
  public final String deploy(String domainId) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Deploying artifact to Mule Runtime");
    }
    String artifactId = doInternalDeploy(artifactResources, runtimeToolingService.get(), domainId, deploymentProperties);
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Application deployed with artifactId: {}", artifactId);
    }
    return artifactId;
  }

  @Override
  public final void dispose(String id) {
    doInternalDispose(id, runtimeToolingService.get());
  }

  protected abstract String doInternalDeploy(ArtifactResources artifactResources, RuntimeToolingService runtimeToolingService,
                                             String domainId, Map<String, String> deploymentProperties);

  protected abstract void doInternalDispose(String id, RuntimeToolingService runtimeToolingService);

  class DeploymentContext {

    private String artifactId;
    private File localLocation;
    private InputStream content;
    private String domainId;
    private Map<String, String> deploymentProperties;
    private RuntimeToolingService runtimeToolingService;

    DeploymentContext(String artifactId, RuntimeToolingService runtimeToolingService) {
      this.artifactId = artifactId;
      this.runtimeToolingService = runtimeToolingService;
    }

    DeploymentContext(InputStream content, String domainId, Map<String, String> deploymentProperties,
                      RuntimeToolingService runtimeToolingService) {
      this.content = content;
      this.domainId = domainId;
      this.runtimeToolingService = runtimeToolingService;
      this.deploymentProperties = deploymentProperties;
    }

    DeploymentContext(File localLocation, String domainId, Map<String, String> deploymentProperties,
                      RuntimeToolingService runtimeToolingService) {
      this.localLocation = localLocation;
      this.domainId = domainId;
      this.runtimeToolingService = runtimeToolingService;
      this.deploymentProperties = deploymentProperties;
    }

    public File getLocalLocation() {
      return localLocation;
    }

    public InputStream getContent() {
      return content;
    }

    public String getDomainId() {
      return domainId;
    }

    public RuntimeToolingService getRuntimeToolingService() {
      return runtimeToolingService;
    }

    public String getArtifactId() {
      return artifactId;
    }

    public Map<String, String> getDeploymentProperties() {
      return deploymentProperties;
    }
  }

}
