/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static org.apache.commons.io.FileUtils.toFile;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import java.io.File;
import java.net.MalformedURLException;
import java.util.List;
import java.util.Optional;

/**
 * {@link ExtensionModelService} that use delegates to {@link InternalExtensionModelService} to resolve {@link ExtensionModel}s
 */
public class DefaultMuleRuntimeExtensionModelProvider extends AbstractMuleRuntimeExtensionModelProvider {

  /**
   * Creates a new instance
   *
   * @param extensionModelService the service to use to load {@link ExtensionModel}s in case of a cache miss.
   */
  public DefaultMuleRuntimeExtensionModelProvider(MavenClient mavenClient, InternalExtensionModelService extensionModelService) {
    super(mavenClient, extensionModelService);
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return extensionModelService.loadExtensionData(getPluginFile(pluginDescriptor))
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<String> getMinMuleVersion(ArtifactDescriptor pluginDescriptor) {
    return extensionModelService.loadExtensionData(getPluginFile(pluginDescriptor))
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(File pluginFile) {
    return extensionModelService.loadExtensionData(pluginFile)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(BundleDependency bundleDependency) {
    return extensionModelService.loadExtensionData(getPluginFile(bundleDependency))
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<String> getMinMuleVersion(File pluginFile) {
    return extensionModelService.loadExtensionData(pluginFile)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<String> getExtensionSchema(File pluginFile) {
    return extensionModelService.loadExtensionData(pluginFile)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public Optional<String> getExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return extensionModelService.loadExtensionData(getPluginFile(pluginDescriptor))
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public List<ExtensionModel> getRuntimeExtensionModels() {
    return extensionModelService.loadRuntimeExtensionModels();
  }

  private File getPluginFile(ArtifactDescriptor pluginDescriptor) {
    return getPluginFile(resolveBundleDependency(toBundleDescriptor(pluginDescriptor)));
  }

  private File getPluginFile(BundleDependency bundleDependency) {
    try {
      return toFile(bundleDependency.getBundleUri().toURL());
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }

}
