/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.datasense;

import static java.util.Optional.ofNullable;
import static org.mule.tooling.client.internal.datasense.ApplicationModelUtils.findComponentWithLocation;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.metadata.ExpressionLanguageMetadataService;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.apikit.metadata.api.MetadataService;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.internal.MetadataProvider;
import org.mule.tooling.client.internal.application.Artifact;
import org.mule.tooling.client.internal.application.ToolingApplicationModel;

import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Represents an artifact, application or domain for DataSense.
 *
 * @since 4.1
 */
public class DataSenseArtifact {

  private Artifact artifact;
  private LazyValue<MetadataCache> metadataCache;
  private MetadataProvider metadataProvider;
  private Map<String, String> artifactProperties;

  private ExpressionLanguageMetadataService expressionLanguageMetadataService;
  private ExpressionLanguage expressionLanguage;
  private MetadataService apikitMetadataService;

  private DataSenseArtifact parent;

  public DataSenseArtifact(Artifact artifact, LazyValue<MetadataCache> metadataCache, MetadataProvider metadataProvider,
                           Map<String, String> artifactProperties,
                           ExpressionLanguageMetadataService expressionLanguageMetadataService,
                           ExpressionLanguage expressionLanguage, MetadataService apikitMetadataService,
                           DataSenseArtifact parent) {
    this.artifact = artifact;
    this.metadataCache = metadataCache;
    this.metadataProvider = metadataProvider;
    this.artifactProperties = artifactProperties;
    this.expressionLanguageMetadataService = expressionLanguageMetadataService;
    this.expressionLanguage = expressionLanguage;
    this.apikitMetadataService = apikitMetadataService;
    this.parent = parent;
  }

  public MetadataCache getMetadataCache() {
    return metadataCache.get();
  }

  public boolean hasComponentModel(Location location) {
    return findComponentWithLocation(artifact.getApplicationModel().getMuleApplicationModel().getRootComponentModel(), location)
        .isPresent();
  }

  public Optional<DataSenseArtifact> getParent() {
    return ofNullable(parent);
  }

  public Map<String, String> getProperties() {
    return artifactProperties;
  }

  public List<ExtensionModel> getExtensionModels() {
    return artifact.getExtensionModels();
  }

  public MetadataProvider getMetadataProvider() {
    return metadataProvider;
  }

  public ToolingApplicationModel getToolingApplicationModel() {
    return artifact.getApplicationModel();
  }

  public ToolingArtifactClassLoader getArtifactClassLoader() {
    return artifact.getArtifactClassLoader();
  }

  public ExpressionLanguageMetadataService getExpressionLanguageMetadataService() {
    return expressionLanguageMetadataService;
  }

  public ExpressionLanguage getExpressionLanguage() {
    return expressionLanguage;
  }

  public MuleVersion getMinMuleVersion() {
    return artifact.getMinMuleVersion();
  }

  @Override
  public String toString() {
    return artifact.toString();
  }

  public MetadataProvider getMetadataProvider(Location location) {
    if (getParent().isPresent() && getParent().get().hasComponentModel(location)) {
      return getParent().get().getMetadataProvider();
    }
    return getMetadataProvider();
  }

  public MetadataService getApikitMetadataService() {
    return apikitMetadataService;
  }
}
