/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.System.getProperty;
import static java.util.Optional.empty;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.copyFile;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.junit.Assume.assumeThat;
import static org.mule.maven.client.internal.util.MavenUtils.getPomModelFromFile;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.maven.client.api.model.RemoteRepository;
import org.mule.maven.client.internal.DefaultLocalRepositorySupplierFactory;

import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.nio.file.Paths;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.maven.model.Model;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

public class ClasspathTestCase {

  private static final String LINE_SEPARATOR = getProperty("line.separator");

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  @Test
  public void toolingRuntimeClientClasspath() throws IOException, URISyntaxException, XmlPullParserException {
    File temporaryRepository = temporaryFolder.newFolder();
    File localRepository = new DefaultLocalRepositorySupplierFactory().environmentMavenRepositorySupplier().get();

    MavenConfiguration mavenConfiguration = MavenConfiguration.newMavenConfigurationBuilder()
        .ignoreArtifactDescriptorRepositories(true)
        .localMavenRepositoryLocation(temporaryRepository)
        .remoteRepository(RemoteRepository.newRemoteRepositoryBuilder().url(localRepository.toURI().toURL()).id("local").build())
        .build();

    File toolingRuntimeClientPomArtifact =
        new File(new File(ClasspathTestCase.class.getProtectionDomain().getCodeSource().getLocation().toURI()).getParentFile()
            .getParentFile(), "pom.xml");
    assumeThat(toolingRuntimeClientPomArtifact.exists(), is(true));

    MavenClientProvider mavenClientProvider = MavenClientProvider.discoverProvider(this.getClass().getClassLoader());
    MavenClient mavenClient = mavenClientProvider.createMavenClient(mavenConfiguration);

    Model model = getPomModelFromFile(toolingRuntimeClientPomArtifact);
    File artifactFolder = temporaryFolder.newFolder();
    File mavenFolder =
        Paths
            .get(artifactFolder.getAbsolutePath(), "META-INF", "maven",
                 model.getGroupId() != null ? model.getGroupId() : model.getParent().getGroupId(), model.getArtifactId())
            .toFile();
    assertThat(mavenFolder.mkdirs(), is(true));
    copyFile(toolingRuntimeClientPomArtifact, new File(mavenFolder, "pom.xml"));

    List<BundleDependency> dependencies =
        mavenClient.resolveArtifactDependencies(artifactFolder, false, true, empty(), empty());

    List<String> expectedClasspath;
    try (InputStreamReader inputStreamReader = new InputStreamReader(
                                                                     this.getClass().getClassLoader()
                                                                         .getResourceAsStream("expected-classpath.txt"))) {
      expectedClasspath = IOUtils.readLines(inputStreamReader);
    }

    List<String> actualClasspath = dependencies.stream()
        .filter(bundleDependency -> bundleDependency.getBundleUri() != null)
        .map(bundleDependency -> new File(bundleDependency.getBundleUri()).getName())
        .collect(toList());

    List<String> missingEntries =
        expectedClasspath.stream().filter(expectedEntry -> !actualClasspath.contains(expectedEntry)).collect(toList());
    List<String> unexpectedEntries =
        actualClasspath.stream().filter(actualEntry -> !expectedClasspath.contains(actualEntry)).collect(toList());

    StringBuilder builder = new StringBuilder();

    if (!missingEntries.isEmpty()) {
      builder.append(LINE_SEPARATOR);
      builder.append("Found Classpath issues on Tooling Runtime Client:");
      builder.append(LINE_SEPARATOR);
      builder.append("********* Missing entries: *********");
      builder.append(LINE_SEPARATOR);
      missingEntries.stream().forEach(entry -> {
        builder.append(entry);
        builder.append(LINE_SEPARATOR);
      });
    }
    if (!unexpectedEntries.isEmpty()) {
      builder.append("********* Unexpected entries: *********");
      builder.append(LINE_SEPARATOR);
      unexpectedEntries.stream().forEach(entry -> {
        builder.append(entry);
        builder.append(" (line: ");
        builder.append(actualClasspath.indexOf(entry));
        builder.append(")");
        builder.append(LINE_SEPARATOR);
      });
    }

    String errorMessage = builder.toString();
    if (errorMessage.length() > 0) {
      fail(errorMessage);
    }
  }
}
