/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.util.Collections.emptyMap;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyObject;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.metadata.MetadataKeyBuilder.newKey;
import static org.mule.runtime.api.metadata.resolving.FailureCode.NOT_AUTHORIZED;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.success;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.metadata.MetadataKey;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.MetadataKeysContainerBuilder;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.dsl.api.component.config.DefaultComponentLocation;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.internal.serialization.Serializer;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.junit.MockitoJUnitRunner;

@Feature("DTOs")
@Story("Metadata Service for Tooling API")
@RunWith(MockitoJUnitRunner.class)
public class ToolingMetadataServiceAdapterTestCase {

  private static final String CATEGORY_NAME = "categoryName";
  private static final String FIRST_KEY_ID = "firstKey";
  private static final String SECOND_KEY_ID = "secondKey";
  private static final String METADATA_RESULT_ERROR_MESSAGE = "Metadata Failure Error";

  @Mock
  private ApplicationModel applicationModel;
  @Mock
  private Set<ExtensionModel> extensionModels;
  @Mock
  private MetadataProvider provider;
  @Mock
  private MetadataCache metadataCache;
  @Mock
  private Serializer serializer;

  private MetadataKeysContainerBuilder builder = MetadataKeysContainerBuilder.getInstance();

  private MetadataKeysRequest metadataKeysRequest;

  @Before
  public void setup() {
    when(metadataCache.getMetadataKeys(any(MetadataCache.MetadataCacheKeyInfo.class), anyObject())).thenAnswer(
                                                                                                               invocationOnMock -> ((Callable<MetadataCache.MetadataResult>) invocationOnMock
                                                                                                                   .getArguments()[1])

                                                                                                                       .call()
                                                                                                                       .getResult());
    metadataKeysRequest = new MetadataKeysRequest();
    final Location location = Location.builder().globalName("globalName").addProcessorsPart().addIndexPart(1).build();
    metadataKeysRequest.setLocation(location);

    ComponentModel globalComponentModel = mock(ComponentModel.class);
    when(applicationModel.findTopLevelNamedComponent(location.getGlobalName())).thenReturn(Optional.of(globalComponentModel));

    DefaultComponentLocation componentLocation = mock(DefaultComponentLocation.class);
    when(globalComponentModel.getComponentLocation()).thenReturn(componentLocation);
    when(componentLocation.getLocation()).thenReturn(location.toString());
  }

  @Test
  public void correctDtoSuccessMetadata() {
    Set<MetadataKey> keys = new LinkedHashSet<>();
    keys.add(newKey(FIRST_KEY_ID).build());
    keys.add(newKey(SECOND_KEY_ID).build());

    MetadataResult<MetadataKeysContainer> successResult = success(builder.add(CATEGORY_NAME, keys).build());
    when(provider.getMetadataKeys(any())).thenReturn(successResult);

    ToolingMetadataServiceAdapter toolingMetadataServiceAdapter =
        new ToolingMetadataServiceAdapter(new LazyValue<>(() -> applicationModel),
                                          new LazyValue<>(() -> extensionModels),
                                          provider, new LazyValue<>(() -> metadataCache),
                                          emptyMap(), serializer);

    org.mule.tooling.client.api.metadata.MetadataResult<org.mule.tooling.client.api.metadata.MetadataKeysContainer> dto =
        toolingMetadataServiceAdapter.getMetadataKeys(metadataKeysRequest);
    assertTrue(dto.isSuccess());

    org.mule.tooling.client.api.metadata.MetadataKeysContainer dtoContainer = dto.get();
    assertThat(dtoContainer.getCategories(), hasSize(1));
    assertThat(dtoContainer.getKeysByCategory().keySet(), hasSize(1));
  }

  @Test
  public void correctDtoFailureMetadata() {
    MetadataResult<MetadataKeysContainer> failure = failure(newFailure()
        .withMessage(METADATA_RESULT_ERROR_MESSAGE)
        .withFailureCode(NOT_AUTHORIZED)
        .withReason(METADATA_RESULT_ERROR_MESSAGE)
        .onKeys());
    when(provider.getMetadataKeys(any())).thenReturn(failure);

    ToolingMetadataServiceAdapter toolingMetadataServiceAdapter =
        new ToolingMetadataServiceAdapter(new LazyValue<>(() -> applicationModel),
                                          new LazyValue<>(() -> extensionModels),
                                          provider, new LazyValue<>(() -> metadataCache),
                                          emptyMap(), serializer);

    org.mule.tooling.client.api.metadata.MetadataResult<org.mule.tooling.client.api.metadata.MetadataKeysContainer> dto =
        toolingMetadataServiceAdapter.getMetadataKeys(metadataKeysRequest);
    assertFalse(dto.isSuccess());

    List<org.mule.tooling.client.api.metadata.MetadataFailure> failures = dto.getFailures();
    assertThat(failures, hasSize(1));
    org.mule.tooling.client.api.metadata.MetadataFailure actualFailure = failures.get(0);
    assertThat(actualFailure.getReason(), is(METADATA_RESULT_ERROR_MESSAGE));
    assertThat(actualFailure.getMessage(), is(METADATA_RESULT_ERROR_MESSAGE));
    assertTrue(actualFailure.getFailingComponent().getValue().equals("KEYS"));
    assertThat(actualFailure.getFailureCode().isNotAuthorized(), is(true));
  }
}
