/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.ResourceLocator;
import org.mule.runtime.config.api.dsl.model.ComponentBuildingDefinitionRegistry;
import org.mule.runtime.config.api.dsl.processor.ArtifactConfig;
import org.mule.runtime.config.internal.ModuleDelegatingEntityResolver;
import org.mule.runtime.config.internal.dsl.model.ClassLoaderResourceProvider;
import org.mule.runtime.config.internal.dsl.xml.XmlNamespaceInfoProviderSupplier;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.core.api.util.xmlsecurity.XMLSecureFactories;
import org.mule.runtime.core.internal.util.DefaultResourceLocator;
import org.mule.runtime.deployment.model.api.DeployableArtifactDescriptor;
import org.mule.runtime.dsl.api.ConfigResource;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;
import org.mule.runtime.dsl.api.xml.parser.ParsingPropertyResolver;
import org.mule.runtime.dsl.api.xml.parser.XmlConfigurationDocumentLoader;
import org.mule.runtime.dsl.api.xml.parser.XmlConfigurationProcessor;
import org.mule.runtime.dsl.api.xml.parser.XmlParsingConfiguration;
import org.mule.tooling.client.api.exception.ToolingException;
import org.xml.sax.EntityResolver;

import javax.xml.parsers.SAXParserFactory;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.runtime.core.internal.config.RuntimeComponentBuildingDefinitionsUtil.getRuntimeComponentBuildingDefinitionProvider;

/**
 * Default implementation for {@link ApplicationModelFactory}.
 *
 * @since 4.0
 */
public class DefaultApplicationModelFactory implements ApplicationModelFactory {

  /**
   * /** {@inheritDoc}
   */
  @Override
  public Optional<ApplicationModel> createApplicationModel(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                           Set<ExtensionModel> extensionModels,
                                                           ClassLoader applicationClassLoader,
                                                           ConfigurationProperties parentConfigurationProperties,
                                                           Map<String, String> deploymentProperties) {
    try {
      return of(loadApplicationModel(deployableArtifactDescriptor, extensionModels, applicationClassLoader,
                                     parentConfigurationProperties, deploymentProperties));
    } catch (Exception e) {
      throw new ToolingException("Error while creating application model", e);
    }
  }


  /**
   * Loads {@link ApplicationModel} from {@link org.mule.runtime.dsl.api.xml.parser.ConfigLine}.
   *
   *
   * @param deployableArtifactDescriptor descriptor for the artifact.
   * @param extensionModels {@link Set} of {@link ExtensionModel extension models} to be used for generating the artifact
   *        model.
   * @param applicationClassLoader {@link ClassLoader} to resolve resources.
   * @param parentConfigurationProperties {@link ConfigurationProperties} for artifact if present.
   * @param deploymentProperties {@link Map} with the deployment properties for the artifact.
   * @return a {@link ApplicationModel}.
   */
  private ApplicationModel loadApplicationModel(
                                                DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                Set<ExtensionModel> extensionModels,
                                                ClassLoader applicationClassLoader,
                                                ConfigurationProperties parentConfigurationProperties,
                                                Map<String, String> deploymentProperties) {
    ToolingArtifactXmlParsingConfiguration parsingConfiguration =
        new ToolingArtifactXmlParsingConfiguration(deployableArtifactDescriptor, extensionModels,
                                                   applicationClassLoader);
    ArtifactConfig artifactConfig = withContextClassLoader(applicationClassLoader, () -> new ArtifactConfig.Builder()
        .setApplicationName(deployableArtifactDescriptor.getName())
        .addConfigFiles(XmlConfigurationProcessor
            .processXmlConfiguration(parsingConfiguration))
        .build());

    return withContextClassLoader(applicationClassLoader,
                                  () -> new ApplicationModel(artifactConfig, null, extensionModels, deploymentProperties,
                                                             of(parentConfigurationProperties),
                                                             of(createComponentBuildingDefinitionRegistry()), false,
                                                             new ClassLoaderResourceProvider(applicationClassLoader)));
  }

  // TODO: EE-5755 - (gfernandes) batch doesn't have an extensionModel therefore use only Mule/MuleEE/Batch component building
  // definition providers to resolve locations and componentTypes
  private static ComponentBuildingDefinitionRegistry createComponentBuildingDefinitionRegistry() {
    final ComponentBuildingDefinitionRegistry componentBuildingDefinitionRegistry =
        new ComponentBuildingDefinitionRegistry();

    getRuntimeComponentBuildingDefinitionProvider()
        .getComponentBuildingDefinitions()
        .forEach(componentBuildingDefinitionRegistry::register);

    return componentBuildingDefinitionRegistry;
  }

  private class ToolingArtifactXmlParsingConfiguration implements XmlParsingConfiguration {

    private Set<ExtensionModel> extensionModels;
    private ConfigResource[] configResources;

    public ToolingArtifactXmlParsingConfiguration(DeployableArtifactDescriptor artifactDescriptor,
                                                  Set<ExtensionModel> extensionModels,
                                                  ClassLoader artifactClassLoader) {
      this.extensionModels = extensionModels;
      this.configResources = artifactDescriptor.getConfigResources().stream()
          .map(configFile -> {
            URL url = artifactClassLoader.getResource(configFile);
            if (url == null) {
              throw new ToolingException(format("Configuration file '%s' cannot be found", configFile));
            }
            try {
              return new ConfigResource(configFile, url.openStream());
            } catch (IOException e) {
              throw new ToolingException(format("Error while reading configuration file '%s'", configFile), e);
            }
          })
          .toArray(ConfigResource[]::new);
    }

    @Override
    public ParsingPropertyResolver getParsingPropertyResolver() {
      return propertyKey -> propertyKey;
    }

    @Override
    public ConfigResource[] getArtifactConfigResources() {
      return this.configResources;
    }

    @Override
    public ResourceLocator getResourceLocator() {
      return new DefaultResourceLocator();
    }

    @Override
    public Supplier<SAXParserFactory> getSaxParserFactory() {
      return () -> XMLSecureFactories.createDefault().getSAXParserFactory();
    }

    @Override
    public XmlConfigurationDocumentLoader getXmlConfigurationDocumentLoader() {
      return XmlConfigurationDocumentLoader.noValidationDocumentLoader();
    }

    @Override
    public EntityResolver getEntityResolver() {
      return new ModuleDelegatingEntityResolver(extensionModels);
    }

    @Override
    public List<XmlNamespaceInfoProvider> getXmlNamespaceInfoProvider() {
      return XmlNamespaceInfoProviderSupplier.createFromExtensionModels(this.extensionModels, empty());
    }
  }
}
