/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.mule.tooling.client.internal.util.Preconditions.checkNotNull;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ComponentModelVisitor;
import org.mule.runtime.api.meta.model.EnrichableModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.function.FunctionModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModel;
import org.mule.runtime.api.meta.model.nested.NestableElementModelVisitor;
import org.mule.runtime.api.meta.model.nested.NestedChainModel;
import org.mule.runtime.api.meta.model.nested.NestedComponentModel;
import org.mule.runtime.api.meta.model.nested.NestedRouteModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.core.api.config.MuleManifest;
import org.mule.runtime.core.api.util.IOUtils;
import org.mule.runtime.extension.api.model.ImmutableExtensionModel;
import org.mule.runtime.extension.api.model.config.ImmutableConfigurationModel;
import org.mule.runtime.extension.api.model.connection.ImmutableConnectionProviderModel;
import org.mule.runtime.extension.api.model.construct.ImmutableConstructModel;
import org.mule.runtime.extension.api.model.function.ImmutableFunctionModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedComponentModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedRouteModel;
import org.mule.runtime.extension.api.model.operation.ImmutableOperationModel;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterGroupModel;
import org.mule.runtime.extension.api.model.parameter.ImmutableParameterModel;
import org.mule.runtime.extension.api.model.source.ImmutableSourceModel;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.runtime.extension.api.property.SinceMuleVersionModelProperty;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;

import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExtensionModelTargetVersionMediator implements MuleRuntimeExtensionModelProvider {

  private static final Logger LOGGER = LoggerFactory.getLogger(ExtensionModelTargetVersionMediator.class);

  private static final String EXTENSION_MODELS = "extension-models";
  private static final String EXTENSION_MODEL_MULE = "extension-model-mule-";
  private MuleVersion targetMuleVersion;
  private MuleRuntimeExtensionModelProvider delegate;
  private List<ExtensionModel> runtimeExtensionModels = new ArrayList<>();

  public ExtensionModelTargetVersionMediator(MuleVersion targetMuleVersion, MuleRuntimeExtensionModelProvider delegate) {
    checkNotNull(targetMuleVersion, "targetMuleVersion cannot be null");
    checkNotNull(delegate, "delegate cannot be null");

    this.targetMuleVersion = targetMuleVersion;
    this.delegate = delegate;

    if (targetMuleVersion.atLeastBase(MuleManifest.getProductVersion())) {
      runtimeExtensionModels.addAll(delegate.getRuntimeExtensionModels());
    } else {
      String versionPatchLess = targetMuleVersion.getMajor() + "." + targetMuleVersion.getMinor();

      if (LOGGER.isInfoEnabled()) {
        LOGGER.info("Using {} version of Mule Runtime Extension models", versionPatchLess);
      }

      ExtensionModelJsonSerializer extensionModelJsonSerializer = new ExtensionModelJsonSerializer();
      runtimeExtensionModels.addAll(readRuntimeExtensionModels(extensionModelJsonSerializer, versionPatchLess));
    }
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return delegate.getExtensionModel(pluginDescriptor).map(extensionModel -> filterExtensionModel(extensionModel));
  }

  @Override
  public Optional<String> getMinMuleVersion(ArtifactDescriptor pluginDescriptor) {
    return delegate.getMinMuleVersion(pluginDescriptor);
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(File plugin) {
    return delegate.getExtensionModel(plugin).map(extensionModel -> filterExtensionModel(extensionModel));
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(BundleDependency bundleDependency) {
    return delegate.getExtensionModel(bundleDependency).map(extensionModel -> filterExtensionModel(extensionModel));
  }

  @Override
  public Optional<String> getMinMuleVersion(File plugin) {
    return delegate.getMinMuleVersion(plugin);
  }

  @Override
  public Optional<String> getExtensionSchema(File plugin) {
    return delegate.getExtensionSchema(plugin);
  }

  @Override
  public Optional<String> getExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return delegate.getExtensionSchema(pluginDescriptor);
  }

  @Override
  public List<ExtensionModel> getRuntimeExtensionModels() {
    return runtimeExtensionModels;
  }

  private List<ExtensionModel> readRuntimeExtensionModels(ExtensionModelJsonSerializer extensionModelJsonSerializer,
                                                          String version) {
    List<ExtensionModel> runtimeExtensionModels = new ArrayList();
    runtimeExtensionModels
        .add(readRuntimeExtensionModel(extensionModelJsonSerializer,
                                       EXTENSION_MODELS + "/" + EXTENSION_MODEL_MULE + version + ".json"));
    runtimeExtensionModels
        .add(readRuntimeExtensionModel(extensionModelJsonSerializer,
                                       EXTENSION_MODELS + "/" + EXTENSION_MODEL_MULE + "ee-" + version + ".json"));
    return runtimeExtensionModels;
  }

  private ExtensionModel readRuntimeExtensionModel(ExtensionModelJsonSerializer extensionModelJsonSerializer, String resource) {

    try (InputStream resourceAsStream = this.getClass().getClassLoader().getResourceAsStream(resource)) {
      if (resourceAsStream == null) {
        throw new RuntimeException(format("Missing extension models resources for runtime version at: %s", resource));
      }
      return extensionModelJsonSerializer
          .deserialize(IOUtils.toString(resourceAsStream));
    } catch (Exception e) {
      throw new RuntimeException("Error while loading extension models for Mule Runtime", e);
    }
  }


  private ExtensionModel filterExtensionModel(ExtensionModel extensionModel) {
    return new ImmutableExtensionModel(extensionModel.getName(),
                                       extensionModel.getDescription(),
                                       extensionModel.getVersion(),
                                       extensionModel.getVendor(),
                                       extensionModel.getCategory(),
                                       filterConfigurationModels(extensionModel.getConfigurationModels()),
                                       filterOperationModels(extensionModel.getOperationModels()),
                                       filterConnectionProviders(extensionModel.getConnectionProviders()),
                                       filterSourceModels(extensionModel.getSourceModels()),
                                       filterFunctionModels(extensionModel.getFunctionModels()),
                                       filterConstructModels(extensionModel.getConstructModels()),
                                       extensionModel.getDisplayModel().orElse(null),
                                       extensionModel.getXmlDslModel(),
                                       extensionModel.getSubTypes(),
                                       extensionModel.getTypes(),
                                       extensionModel.getResources(),
                                       extensionModel.getImportedTypes(),
                                       extensionModel.getErrorModels(),
                                       extensionModel.getExternalLibraryModels(),
                                       extensionModel.getPrivilegedPackages(),
                                       extensionModel.getPrivilegedArtifacts(),
                                       extensionModel.getModelProperties(),
                                       extensionModel.getNotificationModels(),
                                       extensionModel.getDeprecationModel().orElse(null));

  }

  private List<ConfigurationModel> filterConfigurationModels(List<ConfigurationModel> configurationModels) {
    return configurationModels.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(configurationModel -> new ImmutableConfigurationModel(configurationModel.getName(),
                                                                   configurationModel.getDescription(),
                                                                   filterParameterGroupModels(configurationModel
                                                                       .getParameterGroupModels(), targetMuleVersion),
                                                                   filterOperationModels(configurationModel.getOperationModels()),
                                                                   filterConnectionProviders(configurationModel
                                                                       .getConnectionProviders()),
                                                                   filterSourceModels(configurationModel.getSourceModels()),
                                                                   configurationModel.getExternalLibraryModels(),
                                                                   configurationModel.getDisplayModel().orElse(null),
                                                                   configurationModel.getStereotype(),
                                                                   configurationModel.getModelProperties(),
                                                                   configurationModel.getDeprecationModel().orElse(null)))
        .collect(toList());
  }

  private static List<ParameterGroupModel> filterParameterGroupModels(List<ParameterGroupModel> parameterGroupModels,
                                                                      MuleVersion targetMuleVersion) {
    return parameterGroupModels.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(parameterGroupModel -> new ImmutableParameterGroupModel(parameterGroupModel.getName(),
                                                                     parameterGroupModel.getDescription(),
                                                                     filterParameterModels(parameterGroupModel
                                                                         .getParameterModels(), targetMuleVersion),
                                                                     parameterGroupModel.getExclusiveParametersModels(),
                                                                     parameterGroupModel.isShowInDsl(),
                                                                     parameterGroupModel.getDisplayModel().orElse(null),
                                                                     parameterGroupModel.getLayoutModel().orElse(null),
                                                                     parameterGroupModel.getModelProperties()))
        .collect(toList());
  }

  private static List<ParameterModel> filterParameterModels(List<ParameterModel> parameterModels, MuleVersion targetMuleVersion) {
    return parameterModels.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(parameterModel -> new ImmutableParameterModel(parameterModel.getName(),
                                                           parameterModel.getDescription(),
                                                           parameterModel.getType(),
                                                           parameterModel.hasDynamicType(),
                                                           parameterModel.isRequired(),
                                                           parameterModel.isOverrideFromConfig(),
                                                           parameterModel.isComponentId(),
                                                           parameterModel.getExpressionSupport(),
                                                           parameterModel.getDefaultValue(),
                                                           parameterModel.getRole(),
                                                           parameterModel.getDslConfiguration(),
                                                           parameterModel.getDisplayModel().orElse(null),
                                                           parameterModel.getLayoutModel().orElse(null),
                                                           parameterModel.getValueProviderModel().orElse(null),
                                                           parameterModel.getAllowedStereotypes(),
                                                           parameterModel.getModelProperties(),
                                                           parameterModel.getDeprecationModel().orElse(null)))
        .collect(toList());
  }

  @NotNull
  private static Predicate<EnrichableModel> filterBySinceModelProperty(MuleVersion targetMuleVersion) {
    return enrichableModel -> enrichableModel.getModelProperty(SinceMuleVersionModelProperty.class)
        .map(sinceMuleVersionModelProperty -> targetMuleVersion.atLeast(sinceMuleVersionModelProperty.getVersion()))
        .orElse(true);
  }

  private List<OperationModel> filterOperationModels(List<OperationModel> operationModels) {
    return operationModels.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(operationModel -> filterOperationModel(operationModel, targetMuleVersion))
        .collect(toList());
  }

  @NotNull
  private static ImmutableOperationModel filterOperationModel(OperationModel operationModel, MuleVersion targetMuleVersion) {
    return new ImmutableOperationModel(operationModel.getName(),
                                       operationModel.getDescription(),
                                       filterParameterGroupModels(operationModel.getParameterGroupModels(), targetMuleVersion),
                                       filterNestableElementModels(operationModel.getNestedComponents(), targetMuleVersion),
                                       operationModel.getOutput(),
                                       operationModel.getOutputAttributes(),
                                       operationModel.isBlocking(),
                                       operationModel.getExecutionType(),
                                       operationModel.requiresConnection(),
                                       operationModel.isTransactional(),
                                       operationModel.supportsStreaming(),
                                       operationModel.getDisplayModel().orElse(null),
                                       operationModel.getErrorModels(),
                                       operationModel.getStereotype(),
                                       operationModel.getModelProperties(),
                                       operationModel.getNotificationModels(),
                                       operationModel.getDeprecationModel().orElse(null));
  }

  private static List<? extends NestableElementModel> filterNestableElementModels(List<? extends NestableElementModel> components,
                                                                                  MuleVersion targetMuleVersion) {
    return components.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(model -> filterNestedComponentModel(model, targetMuleVersion))
        .collect(toList());
  }

  private static NestableElementModel filterNestedComponentModel(NestableElementModel nestedModel,
                                                                 MuleVersion targetMuleVersion) {
    final NestableElementModelFilterVisitor nestableElementModelVisitor =
        new NestableElementModelFilterVisitor(targetMuleVersion);
    nestedModel.accept(nestableElementModelVisitor);
    return nestableElementModelVisitor.getNestableElementModelFiltered();
  }

  private static class NestableElementModelFilterVisitor implements NestableElementModelVisitor {

    private NestableElementModel nestableElementModel = null;
    private MuleVersion targetMuleVersion;

    public NestableElementModelFilterVisitor(MuleVersion targetMuleVersion) {
      this.targetMuleVersion = targetMuleVersion;
    }

    public NestableElementModel getNestableElementModelFiltered() {
      return nestableElementModel;
    }

    @Override
    public void visit(NestedComponentModel nestedComponentModel) {
      nestableElementModel = new ImmutableNestedComponentModel(nestedComponentModel.getName(),
                                                               nestedComponentModel.getDescription(),
                                                               nestedComponentModel.getDisplayModel().orElse(null),
                                                               nestedComponentModel.isRequired(),
                                                               nestedComponentModel.getAllowedStereotypes(),
                                                               nestedComponentModel.getModelProperties());
    }

    @Override
    public void visit(NestedChainModel nestedChainModel) {
      nestableElementModel = new ImmutableNestedComponentModel(nestedChainModel.getName(),
                                                               nestedChainModel.getDescription(),
                                                               nestedChainModel.getDisplayModel().orElse(null),
                                                               nestedChainModel.isRequired(),
                                                               nestedChainModel.getAllowedStereotypes(),
                                                               nestedChainModel.getModelProperties());
    }

    @Override
    public void visit(NestedRouteModel nestedRouteModel) {
      nestableElementModel = new ImmutableNestedRouteModel(nestedRouteModel.getName(),
                                                           nestedRouteModel.getDescription(),
                                                           filterParameterGroupModels(nestedRouteModel.getParameterGroupModels(),
                                                                                      targetMuleVersion),
                                                           nestedRouteModel.getDisplayModel().orElse(null),
                                                           nestedRouteModel.getMinOccurs(),
                                                           nestedRouteModel.getMaxOccurs().orElse(null),
                                                           filterNestableElementModels(nestedRouteModel.getNestedComponents(),
                                                                                       targetMuleVersion),
                                                           nestedRouteModel.getModelProperties());
    }
  }


  private List<ConnectionProviderModel> filterConnectionProviders(List<ConnectionProviderModel> connectionProviders) {
    return connectionProviders.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(connectionProviderModel -> new ImmutableConnectionProviderModel(connectionProviderModel.getName(),
                                                                             connectionProviderModel.getDescription(),
                                                                             filterParameterGroupModels(connectionProviderModel
                                                                                 .getParameterGroupModels(), targetMuleVersion),
                                                                             connectionProviderModel
                                                                                 .getConnectionManagementType(),
                                                                             connectionProviderModel
                                                                                 .supportsConnectivityTesting(),
                                                                             connectionProviderModel.getExternalLibraryModels(),
                                                                             connectionProviderModel.getDisplayModel()
                                                                                 .orElse(null),
                                                                             connectionProviderModel.getStereotype(),
                                                                             connectionProviderModel.getModelProperties(),
                                                                             connectionProviderModel.getDeprecationModel()
                                                                                 .orElse(null)))
        .collect(toList());
  }

  private List<SourceModel> filterSourceModels(List<SourceModel> sourceModels) {
    return sourceModels.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(sourceModel -> filterSourceModel(sourceModel, targetMuleVersion))
        .collect(toList());
  }

  @NotNull
  private static ImmutableSourceModel filterSourceModel(SourceModel sourceModel, MuleVersion targetMuleVersion) {
    return new ImmutableSourceModel(sourceModel.getName(),
                                    sourceModel.getDescription(),
                                    sourceModel.hasResponse(),
                                    sourceModel.runsOnPrimaryNodeOnly(),
                                    filterParameterGroupModels(sourceModel.getParameterGroupModels(), targetMuleVersion),
                                    filterNestableElementModels(sourceModel.getNestedComponents(), targetMuleVersion),
                                    sourceModel.getOutput(),
                                    sourceModel.getOutputAttributes(),
                                    sourceModel.getSuccessCallback(),
                                    sourceModel.getErrorCallback(),
                                    sourceModel.getTerminateCallback(),
                                    sourceModel.requiresConnection(),
                                    sourceModel.isTransactional(),
                                    sourceModel.supportsStreaming(),
                                    sourceModel.getDisplayModel().orElse(null),
                                    sourceModel.getStereotype(),
                                    sourceModel.getErrorModels(), sourceModel.getModelProperties(),
                                    sourceModel.getNotificationModels(),
                                    sourceModel.getDeprecationModel().orElse(null));
  }

  private List<FunctionModel> filterFunctionModels(List<FunctionModel> functionModels) {
    return functionModels.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(functionModel -> new ImmutableFunctionModel(functionModel.getName(),
                                                         functionModel.getDescription(),
                                                         filterParameterGroupModels(functionModel.getParameterGroupModels(),
                                                                                    targetMuleVersion),
                                                         functionModel.getOutput(),
                                                         functionModel.getDisplayModel().orElse(null),
                                                         functionModel.getModelProperties(),
                                                         functionModel.getDeprecationModel().orElse(null)))
        .collect(toList());
  }

  private List<ConstructModel> filterConstructModels(List<ConstructModel> constructModels) {
    return constructModels.stream()
        .filter(filterBySinceModelProperty(targetMuleVersion))
        .map(constructModel -> filterConstructModel(constructModel, targetMuleVersion))
        .collect(toList());
  }

  @NotNull
  private static ImmutableConstructModel filterConstructModel(ConstructModel constructModel, MuleVersion targetMuleVersion) {
    return new ImmutableConstructModel(constructModel.getName(),
                                       constructModel.getDescription(),
                                       filterParameterGroupModels(constructModel.getParameterGroupModels(), targetMuleVersion),
                                       filterNestableElementModels(constructModel.getNestedComponents(), targetMuleVersion),
                                       constructModel.allowsTopLevelDeclaration(),
                                       constructModel.getDisplayModel().orElse(null),
                                       constructModel.getErrorModels(),
                                       constructModel.getStereotype(),
                                       constructModel.getModelProperties(),
                                       constructModel.getDeprecationModel().orElse(null));
  }

  public static class ComponentModelMediator<T extends ComponentModel> {

    private T component;
    private MuleVersion targetMuleVersion;

    public ComponentModelMediator(MuleVersion targetMuleVersion, T component) {
      checkNotNull(targetMuleVersion, "targetMuleVersion cannot be null");
      checkNotNull(component, "component cannot be null");

      this.targetMuleVersion = targetMuleVersion;
      this.component = component;
    }

    public T getFilteredComponentModel() {
      Reference<T> filteredModel = new Reference<>();
      component.accept(new ComponentModelVisitor() {

        @Override
        public void visit(ConstructModel constructModel) {
          filteredModel.set((T) filterConstructModel(constructModel, targetMuleVersion));
        }

        @Override
        public void visit(OperationModel operationModel) {
          filteredModel.set((T) filterOperationModel(operationModel, targetMuleVersion));
        }

        @Override
        public void visit(SourceModel sourceModel) {
          filteredModel.set((T) filterSourceModel(sourceModel, targetMuleVersion));
        }
      });

      return filteredModel.get();

    }
  }

}
