/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.client.WireMock.urlEqualTo;
import static com.github.tomakehurst.wiremock.core.WireMockConfiguration.wireMockConfig;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.not;
import static org.hamcrest.Matchers.nullValue;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.tooling.client.internal.application.ArtifactResources;

import com.github.tomakehurst.wiremock.junit.WireMockClassRule;

import java.io.File;
import java.net.URI;
import java.net.URL;

import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class ArtifactResourcesTestCase {

  private static final String APPLICATIONS_DATASENSE_STATIC = "applications/datasense-static";
  private static final String JAR_EXTENSION = ".jar";
  @Rule
  public WireMockClassRule wireMockRule = new WireMockClassRule(wireMockConfig().dynamicPort());
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public ExpectedException expectedException = none();

  private File artifactFile;
  private File artifactJarFile;
  private URL artifactContentUrl;

  private final String REQUEST_PATH = "/app";

  @Before
  public void before() throws Exception {
    artifactFile = toFile(this.getClass().getClassLoader().getResource(APPLICATIONS_DATASENSE_STATIC));
    artifactJarFile = toFile(this.getClass().getClassLoader().getResource(APPLICATIONS_DATASENSE_STATIC + JAR_EXTENSION));
    artifactContentUrl = artifactFile.toURI().toURL();
  }

  @Test
  public void createArtifactResourcesFromRemoteUrl() throws Exception {
    File jarFile = temporaryFolder.newFile();
    compress(jarFile, artifactFile);

    wireMockRule.stubFor(
                         get(urlEqualTo("/app"))
                             .willReturn(aResponse()
                                 .withStatus(200)
                                 .withHeader("Content-Type", "application/octet-stream")
                                 .withBody(IOUtils.toByteArray(jarFile.toURI().toURL()))));
    int port = wireMockRule.port();

    artifactContentUrl = new URI("http://localhost:" + port + REQUEST_PATH).toURL();

    ArtifactResources artifactResources = new ArtifactResources(artifactContentUrl);

    assertThat(artifactResources.getArtifactUrlContent(), equalTo(artifactContentUrl));
    assertThat(artifactResources.getArtifactType(), equalTo(ArtifactType.APP));
    assertThat(artifactResources.isRemote(), is(true));

    assertThat(artifactResources.getRootArtifactFile(), not(nullValue()));
    assertThat(artifactResources.getRootArtifactFile().exists(), is(true));

    assertThat(artifactResources.getWorkingDirectory(), not(nullValue()));
    assertThat(artifactResources.getWorkingDirectory().exists(), is(true));

    assertThat(artifactResources.getRootArtifactFile(), not(artifactJarFile));

    assertThat(artifactResources.getRemoteFileDownloaded(), not(nullValue()));

    File rootArtifactFile = artifactResources.getRootArtifactFile();
    File workingDirectory = artifactResources.getWorkingDirectory();
    File remoteFileDownloaded = artifactResources.getRemoteFileDownloaded();
    artifactResources.dispose();

    assertThat(rootArtifactFile.exists(), is(false));
    assertThat(workingDirectory.exists(), is(false));
    assertThat(remoteFileDownloaded.exists(), is(false));
  }

  @Test
  public void createArtifactResourcesFromLocalFileExplodedUrl() throws Exception {
    ArtifactResources artifactResources = new ArtifactResources(artifactContentUrl);

    assertThat(artifactResources.getArtifactUrlContent(), equalTo(artifactContentUrl));
    assertThat(artifactResources.getArtifactType(), equalTo(ArtifactType.APP));
    assertThat(artifactResources.isRemote(), is(false));
    assertThat(artifactResources.getRemoteFileDownloaded(), nullValue());

    assertThat(artifactResources.getRootArtifactFile(), not(nullValue()));
    assertThat(artifactResources.getRootArtifactFile().exists(), is(true));
    assertThat(artifactResources.getRootArtifactFile(), equalTo(artifactFile));

    assertThat(artifactResources.getWorkingDirectory(), not(nullValue()));
    assertThat(artifactResources.getWorkingDirectory().exists(), is(true));

    File rootArtifactFile = artifactResources.getRootArtifactFile();
    File workingDirectory = artifactResources.getWorkingDirectory();
    artifactResources.dispose();

    assertThat(rootArtifactFile.exists(), is(true));
    assertThat(workingDirectory.exists(), is(false));
  }

  @Test
  public void createArtifactResourcesFromLocalFilePackagedUrl() throws Exception {
    artifactContentUrl = artifactJarFile.toURI().toURL();
    ArtifactResources artifactResources = new ArtifactResources(artifactContentUrl);

    assertThat(artifactResources.getArtifactUrlContent(), equalTo(artifactContentUrl));
    assertThat(artifactResources.getArtifactType(), equalTo(ArtifactType.APP));
    assertThat(artifactResources.isRemote(), is(false));
    assertThat(artifactResources.getRemoteFileDownloaded(), nullValue());

    assertThat(artifactResources.getRootArtifactFile(), not(nullValue()));
    assertThat(artifactResources.getRootArtifactFile().exists(), is(true));
    assertThat(artifactResources.getRootArtifactFile(), not(artifactFile));

    assertThat(artifactResources.getWorkingDirectory(), not(nullValue()));
    assertThat(artifactResources.getWorkingDirectory().exists(), is(true));

    File rootArtifactFile = artifactResources.getRootArtifactFile();
    File workingDirectory = artifactResources.getWorkingDirectory();
    artifactResources.dispose();

    assertThat(rootArtifactFile.exists(), is(false));
    assertThat(workingDirectory.exists(), is(false));
  }

}
