/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Throwables.propagateIfPossible;
import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.DataSenseResolveFactory.fromDataSenseResolutionScopeDTO;
import static org.mule.tooling.client.internal.LocationFactory.fromLocationDTO;
import org.mule.datasense.api.DataSense;
import org.mule.datasense.api.DataSenseInfo;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.CachedDataSenseProvider;
import org.mule.datasense.api.metadataprovider.DataSenseConfiguration;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataCacheProvider;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.api.metadataprovider.DefaultDataSenseProvider;
import org.mule.datasense.api.metadataprovider.ExtensionsProvider;
import org.mule.datasense.impl.DefaultDataSense;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.tooling.client.api.datasense.ApplicationResolutionScope;
import org.mule.tooling.client.api.datasense.ComponentResolutionScope;
import org.mule.tooling.client.api.datasense.ConfigResolutionScope;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.application.ToolingApplicationModel;
import org.mule.tooling.client.internal.datasense.DataSenseArtifact;
import org.mule.tooling.client.internal.datasense.DataSenseResolutionScopeVisitor;
import org.mule.tooling.client.internal.serialization.Serializer;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link DataSenseService}.
 *
 * @since 4.0
 */
public class DefaultDataSenseService implements DataSenseService, Command {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String APP_TYPES_DATA = "application-types.xml";
  private static final String APP_TYPES_TEST_DATA = "application-types-test.xml";

  private DataSense dataSense = new DefaultDataSense();

  private DataSenseArtifact dataSenseArtifact;

  private AtomicBoolean disposed = new AtomicBoolean(false);

  private Serializer serializer;

  private DataSensePartsFactory dataSensePartsFactory;

  /**
   * Creates an instance of the DataSense service.
   *
   * @param dataSenseArtifact {@link DataSenseArtifact} to resolve DataSense. Non null.
   * @param serializer {@link Serializer} to resolve the serialization from API invocations. Non null.
   */
  public DefaultDataSenseService(DataSenseArtifact dataSenseArtifact, Serializer serializer) {
    checkNotNull(serializer, "serializer cannot be null");
    checkNotNull(dataSenseArtifact, "dataSenseArtifact cannot be null");

    this.serializer = serializer;
    this.dataSenseArtifact = dataSenseArtifact;

    this.dataSensePartsFactory = new DataSensePartsFactory();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<org.mule.tooling.client.api.datasense.DataSenseInfo> resolveDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    return dataSensePartsFactory.toDataSenseInfoDTO(internalResolveDataSense(dataSenseRequest));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<DataSenseComponentInfo> resolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    return dataSensePartsFactory.toDataSenseComponentInfoDTO(internalResolveComponentDataSense(dataSenseRequest));
  }

  private Optional<DataSenseInfo> internalResolveDataSense(DataSenseRequest dataSenseRequest) {
    checkState();

    Location location = fromLocationDTO(dataSenseRequest.getLocation());

    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = new DataSenseMetadataCacheAdapter(dataSenseArtifact);

    if (logger.isDebugEnabled()) {
      logger.debug(format("Resolving DataSense for component location: %s on %s", dataSenseRequest.getLocation(),
                          dataSenseArtifact));
    }

    try {
      ApplicationModel applicationModel = buildDataSenseApplicationModel(dataSenseArtifact.getToolingApplicationModel());

      DataSenseProvider dataSenseProvider =
          getDataSenseProvider(applicationModel,
                               () -> ImmutableSet.<ExtensionModel>builder().addAll(dataSenseArtifact.getExtensionModels())
                                   .build(),
                               of(dataSenseMetadataCacheProvider),
                               dataSenseRequest);


      return withContextClassLoader(dataSenseArtifact.getArtifactClassLoader().getClassLoader(),
                                    () -> this.dataSense.resolve(fromDataSenseResolutionScopeDTO(dataSenseRequest
                                        .getDataSenseResolutionScope()), applicationModel,
                                                                 dataSenseProvider));
    } catch (Exception e) {
      propagateIfPossible(e, MissingToolingConfigurationException.class);
      throw new ToolingException(format("Error while resolving DataSense for location: %s on %s", location, dataSenseArtifact),
                                 e);
    }
  }

  private Optional<org.mule.datasense.api.DataSenseComponentInfo> internalResolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    Location location = fromLocationDTO(dataSenseRequest.getLocation());
    checkNotNull(location, "location cannot be null");

    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = new DataSenseMetadataCacheAdapter(dataSenseArtifact);

    if (logger.isDebugEnabled()) {
      logger.debug(format("Resolving DataSense for location: %s on %s", dataSenseRequest.getLocation(), dataSenseArtifact));
    }

    try {
      ApplicationModel applicationModel = buildDataSenseApplicationModel(dataSenseArtifact.getToolingApplicationModel());
      DataSenseProvider dataSenseProvider =
          getDataSenseProvider(applicationModel,
                               () -> ImmutableSet.<ExtensionModel>builder().addAll(dataSenseArtifact.getExtensionModels())
                                   .build(),
                               of(dataSenseMetadataCacheProvider),
                               dataSenseRequest);

      return withContextClassLoader(dataSenseArtifact.getArtifactClassLoader().getClassLoader(),
                                    () -> this.dataSense.resolveComponent(location, applicationModel, dataSenseProvider));
    } catch (Exception e) {
      propagateIfPossible(e, MissingToolingConfigurationException.class);
      throw new ToolingException(format("Error while resolving DataSense for location %s on %s", location, dataSenseArtifact), e);
    }
  }

  public void dispose() {
    disposed.compareAndSet(false, true);
  }

  private ApplicationModel buildDataSenseApplicationModel(ToolingApplicationModel toolingApplicationModel) {
    List<String> typesDataList = new ArrayList<>();
    typesDataList.addAll(getApplicationTypes(APP_TYPES_DATA, dataSenseArtifact));
    typesDataList.addAll(getApplicationTypes(APP_TYPES_TEST_DATA, dataSenseArtifact));
    return new DefaultApplicationModel(toolingApplicationModel, typesDataList);
  }

  private List<String> getApplicationTypes(String resource, DataSenseArtifact dataSenseArtifact) {
    List<String> applicationTypes = new ArrayList<>();
    try {
      final Enumeration<URL> resources = dataSenseArtifact.getArtifactClassLoader().getClassLoader().getResources(resource);
      while (resources.hasMoreElements()) {
        URL url = resources.nextElement();
        if (logger.isDebugEnabled()) {
          logger.debug("Found application custom types data from: " + url);
        }
        try {
          applicationTypes.add(IOUtils.toString(url));
        } catch (IOException e) {
          throw new ToolingException(format("Error while reading application custom types file: %s for %s", url,
                                            dataSenseArtifact),
                                     e);
        }
      }

    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
    return applicationTypes;
  }

  private DataSenseProvider getDataSenseProvider(ApplicationModel applicationModel, ExtensionsProvider extensionsProvider,
                                                 Optional<DataSenseMetadataCacheProvider> dataSenseMetadataCacheProvider,
                                                 DataSenseRequest request) {
    DataSenseProvider dataSenseProvider =
        new DefaultDataSenseProvider(extensionsProvider, new ToolingDataSenseMetadataProvider(dataSenseArtifact,
                                                                                              request.getMetadataTimeout()),
                                     getDataSenseConfiguration(request),
                                     dataSenseArtifact.getExpressionLanguageMetadataService(),
                                     dataSenseArtifact.getExpressionLanguage(),
                                     dataSenseArtifact.getApikitMetadataService().getApikitMetadataBuilder());
    if (dataSenseMetadataCacheProvider.isPresent()) {
      dataSenseProvider = new CachedDataSenseProvider(applicationModel, dataSenseMetadataCacheProvider.get(), dataSenseProvider);
    }

    return dataSenseProvider;
  }

  private DataSenseConfiguration getDataSenseConfiguration(DataSenseRequest request) {
    DataSenseConfiguration dataSenseConfiguration = new DataSenseConfiguration();
    request.getDataSenseResolutionScope().accept(new DataSenseConfigurationResolutionScopeVisitor(dataSenseConfiguration));
    return dataSenseConfiguration;
  }

  private class DataSenseConfigurationResolutionScopeVisitor implements DataSenseResolutionScopeVisitor {

    private DataSenseConfiguration dataSenseConfiguration;

    public DataSenseConfigurationResolutionScopeVisitor(DataSenseConfiguration dataSenseConfiguration) {
      this.dataSenseConfiguration = dataSenseConfiguration;
    }

    @Override
    public void visit(ApplicationResolutionScope resolutionScope) {}

    @Override
    public void visit(ConfigResolutionScope resolutionScope) {}

    @Override
    public void visit(ComponentResolutionScope resolutionScope) {
      if (resolutionScope.keyEnrichment().isEnabled()) {
        dataSenseConfiguration.setKeyEnrichment(resolutionScope.keyEnrichment().get());
      }
    }

  }

  private void checkState() {
    Preconditions.checkState(!disposed.get(), "Service already disposed, cannot be used anymore");
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "resolveDataSense": {
        org.mule.runtime.api.util.Preconditions.checkState(arguments.length == 1,
                                                           format("Wrong number of arguments when invoking method created on %s",
                                                                  this.getClass().getName()));
        org.mule.runtime.api.util.Preconditions
            .checkState(classes.length == 1 && classes[0].equals(DataSenseRequest.class.getName()),
                        format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(resolveDataSense(serializer.deserialize(arguments[0])));
      }
      case "resolveComponentDataSense": {
        org.mule.runtime.api.util.Preconditions.checkState(arguments.length == 1,
                                                           format("Wrong number of arguments when invoking method created on %s",
                                                                  this.getClass().getName()));
        org.mule.runtime.api.util.Preconditions
            .checkState(classes.length == 1 && classes[0].equals(DataSenseRequest.class.getName()),
                        format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(resolveComponentDataSense(serializer.deserialize(arguments[0])));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

  class DefaultApplicationModel implements ApplicationModel {

    private final ComponentModel rootComponentModel;
    private final ToolingApplicationModel toolingApplicationModel;
    private final List<String> typesData;

    DefaultApplicationModel(ToolingApplicationModel toolingApplicationModel, List<String> typesData) {
      this.toolingApplicationModel = toolingApplicationModel;
      this.typesData = typesData;

      List<ComponentModel> innerComponents = new ArrayList<>();
      toolingApplicationModel.getParent().ifPresent(parent -> innerComponents
          .addAll(parent.getMuleApplicationModel().getRootComponentModel().getInnerComponents()));
      innerComponents.addAll(toolingApplicationModel.getMuleApplicationModel().getRootComponentModel().getInnerComponents());

      ComponentModel.Builder builder = new ComponentModel.Builder();
      innerComponents.forEach(componentModel -> builder.addChildComponentModel(componentModel));
      builder.markAsRootComponent();
      builder.setIdentifier(toolingApplicationModel.getMuleApplicationModel().getRootComponentModel().getIdentifier());

      toolingApplicationModel.getMuleApplicationModel().getRootComponentModel().getCustomAttributes().forEach((name, value) -> {
        builder.addCustomAttribute(name, value);
      });
      rootComponentModel = builder.build();
    }

    @Override
    public ComponentModel findRootComponentModel() {
      return rootComponentModel;
    }

    @Override
    public Optional<ComponentModel> findNamedComponent(String name) {
      Optional<ComponentModel> found = empty();
      if (toolingApplicationModel.getParent().isPresent()) {
        found = toolingApplicationModel.getParent().get().getMuleApplicationModel().findNamedElement(name);
      }
      if (found.isPresent()) {
        return found;
      }

      return toolingApplicationModel.getMuleApplicationModel().findNamedElement(name);
    }

    @Override
    public List<String> findTypesDataList() {
      return this.typesData;
    }

    public Optional<URI> findResource(String resource) {
      return this.toolingApplicationModel.findResource(resource);
    }

    @Override
    public org.mule.runtime.config.internal.model.ApplicationModel getMuleApplicationModel() {
      return toolingApplicationModel.getMuleApplicationModel();
    }

  }

}
