/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import static java.util.UUID.randomUUID;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.APP;
import static org.mule.runtime.core.api.config.bootstrap.ArtifactType.DOMAIN;
import static org.mule.runtime.module.deployment.impl.internal.maven.AbstractMavenClassLoaderModelLoader.CLASSLOADER_MODEL_MAVEN_REACTOR_RESOLVER;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleArtifactLoaderDescriptor;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.domain.DomainDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.tooling.internal.DefaultToolingService;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.icons.ExtensionIconsService;
import org.mule.tooling.client.api.message.history.MessageHistoryService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.Artifact;
import org.mule.tooling.client.internal.application.ArtifactResources;
import org.mule.tooling.client.internal.application.DefaultApplication;
import org.mule.tooling.client.internal.application.DefaultDomain;
import org.mule.tooling.client.internal.application.Domain;
import org.mule.tooling.client.internal.artifact.DefaultArtifactSerializationService;
import org.mule.tooling.client.internal.dsl.DefaultDslSyntaxResolverService;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;
import org.mule.tooling.client.internal.icons.DefaultExtensionIconsService;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.service.ServiceRegistry;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * Implementation of a {@link ToolingRuntimeClient}. It uses SPI to look up for {@link RuntimeToolingService} implementation to
 * resolve tooling operations.
 *
 * @since 4.0
 */
public class DefaultToolingRuntimeClient implements ToolingRuntimeClient, Command {

  private DefaultToolingArtifactContext context;

  private ExtensionModelService extensionModelService;
  private ArtifactSerializationService artifactSerializationService;
  private DslSyntaxResolverService dslSyntaxResolverService;
  private MessageHistoryService messageHistoryService;
  private ExtensionIconsService extensionIconsService;

  /**
   * Creates an instance of the client.
   *
   * @param mavenClient {@link MavenClient} to resolve dependencies. Non null.
   * @param serializer {@link Serializer} to serialize objects from/to API communication. Non null.
   * @param agentConfigurationOptional {@link Optional} Mule Agent configuration. Non null.
   * @param muleRuntimeExtensionModelProvider provider to load {@link org.mule.runtime.api.meta.model.ExtensionModel}s. Non null.
   * @param muleArtifactResourcesRegistry {@link MuleArtifactResourcesRegistry} to access the artifact registry. Non null.
   * @param applicationCache {@link ApplicationCache} to fetch or create {@link org.mule.tooling.client.internal.application.Application}. Non null.
   * @param domainCache {@link DomainCache} to fetch or create {@link org.mule.tooling.client.internal.application.Domain}. Non null.
   * @param metadataCacheFactoryOptional {@link Optional} {@link MetadataCacheFactory} to be used when resolving Metadata during data sense resolution. Non null.
   * @param dslSyntaxServiceCache {@link DslSyntaxServiceCache}. Non null.
   */
  public DefaultToolingRuntimeClient(MavenClient mavenClient,
                                     Serializer serializer,
                                     Optional<AgentConfiguration> agentConfigurationOptional,
                                     MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                     MuleArtifactResourcesRegistry muleArtifactResourcesRegistry,
                                     ApplicationCache applicationCache,
                                     DomainCache domainCache, Optional<MetadataCacheFactory> metadataCacheFactoryOptional,
                                     DslSyntaxServiceCache dslSyntaxServiceCache,
                                     ServiceRegistry serviceRegistry) {
    checkNotNull(mavenClient, "aetherDependencyResolver cannot be null");
    checkNotNull(serializer, "serializer cannot be null");
    checkNotNull(agentConfigurationOptional, "agentConfigurationOptional cannot be null");
    checkNotNull(muleRuntimeExtensionModelProvider, "muleRuntimeExtensionModelProvider cannot be null");
    checkNotNull(muleArtifactResourcesRegistry, "muleArtifactResourcesRegistry cannot be null");
    checkNotNull(applicationCache, "applicationCache cannot be null");
    checkNotNull(domainCache, "domainCache cannot be null");
    checkNotNull(metadataCacheFactoryOptional, "metadataCacheFactoryOptional cannot be null");
    checkNotNull(dslSyntaxServiceCache, "dslSyntaxServiceCache cannot be null");
    checkNotNull(serviceRegistry, "serviceRegistry cannot be null");

    context = new DefaultToolingArtifactContext(muleArtifactResourcesRegistry);
    context.setAgentConfiguration(agentConfigurationOptional);
    context.setSerializer(serializer);
    context.setMavenClient(mavenClient);
    context.setMuleRuntimeExtensionModelProvider(muleRuntimeExtensionModelProvider);
    context.setApplicationCache(applicationCache);
    context.setDomainCache(domainCache);
    context.setMetadataCacheFactory(metadataCacheFactoryOptional);
    context.setServiceRegistry(serviceRegistry);

    extensionModelService = new ToolingExtensionModelAdapter(muleRuntimeExtensionModelProvider, context.getSerializer(),
                                                             muleArtifactResourcesRegistry.getTargetMuleVersion());
    artifactSerializationService =
        new DefaultArtifactSerializationService(muleRuntimeExtensionModelProvider.getRuntimeExtensionModels(),
                                                muleRuntimeExtensionModelProvider,
                                                mavenClient, context.getSerializer());
    messageHistoryService =
        new DefaultMessageHistoryService(new LazyValue<>(() -> context.getRuntimeToolingService()), context.getSerializer());
    extensionIconsService = new DefaultExtensionIconsService(context.getMavenClient(), context.getSerializer());
    dslSyntaxResolverService =
        new DefaultDslSyntaxResolverService(dslSyntaxServiceCache, mavenClient, muleRuntimeExtensionModelProvider,
                                            context.getSerializer());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExtensionModelService extensionModelService() {
    return extensionModelService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactSerializationService artifactSerializationService() {
    return artifactSerializationService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DslSyntaxResolverService dslSyntaxResolverService() {
    return dslSyntaxResolverService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageHistoryService messageHistoryService() {
    return messageHistoryService;
  }

  @Override
  public ExtensionIconsService iconsService() {
    return extensionIconsService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> toolingArtifactProperties) {
    String id = randomUUID().toString();

    ArtifactResources artifactResources = new ArtifactResources(artifactUrlContent);
    if (artifactResources.getArtifactType().equals(DOMAIN)) {
      Domain domain = createDomain(id, artifactResources, toolingArtifactProperties);
      LazyValue<MetadataCache> metadataCache = createMetadataCache(id, domain);
      return newToolingArtifact(id, context.getDomainCache(), metadataCache, domain, context.getSerializer(),
                                context.getServiceRegistry());
    } else if (artifactResources.getArtifactType().equals(APP)) {
      ApplicationDescriptor applicationDescriptor;
      try {
        applicationDescriptor =
            context.getApplicationDescriptorFactory().create(artifactResources.getRootArtifactFile(), empty());
      } catch (ToolingException e) {
        throw e;
      } catch (Exception e) {
        throw new ToolingException(format("Error while creating application from url: %s", artifactUrlContent), e);
      }
      ToolingArtifact domainToolingArtifact = null;
      final Reference<Domain> domainReference = new Reference<>();
      LazyValue<MetadataCache> parentMetadataCache = null;
      if (applicationDescriptor.getDomainDescriptor().isPresent()) {
        MavenClient mavenClient = context.getMavenClient();
        BundleDescriptor domainBundleDescriptor = applicationDescriptor.getDomainDescriptor().get();
        BundleDependency domainBundleDependency =
            mavenClient.resolveBundleDescriptor(new org.mule.maven.client.api.model.BundleDescriptor.Builder()
                .setGroupId(domainBundleDescriptor.getGroupId())
                .setArtifactId(domainBundleDescriptor.getArtifactId())
                .setVersion(domainBundleDescriptor.getVersion())
                .setClassifier(domainBundleDescriptor.getClassifier().get())
                .setType(domainBundleDescriptor.getType())
                .build());
        String domainId = randomUUID().toString();
        try {
          domainReference.set(createDomain(domainId, new ArtifactResources(domainBundleDependency.getBundleUri().toURL()),
                                           toolingArtifactProperties));
        } catch (MalformedURLException e) {
          throw new ToolingException("Error while getting Mule domainReference artifact url", e);
        }
        parentMetadataCache = createMetadataCache(domainId, domainReference.get());
        domainToolingArtifact =
            newToolingArtifact(domainId, context.getDomainCache(), parentMetadataCache, domainReference.get(),
                               context.getSerializer(), context.getServiceRegistry());
      }
      try {
        ApplicationCache applicationCache = context.getApplicationCache();
        Application application =
            applicationCache.getApplication(id,
                                            () -> new DefaultApplication(id, artifactResources, applicationDescriptor,
                                                                         domainReference.get(),
                                                                         context, toolingArtifactProperties,
                                                                         domainReference.get() != null));
        LazyValue<MetadataCache> metadataCache = createMetadataCache(id, application);
        return newToolingArtifact(id, context.getApplicationCache(), metadataCache, application, domainToolingArtifact,
                                  parentMetadataCache, context.getSerializer(), context.getServiceRegistry());
      } catch (ToolingException e) {
        disposeIfNeeded(domainToolingArtifact);
        throw e;
      } catch (Exception e) {
        disposeIfNeeded(domainToolingArtifact);
        throw new ToolingException(format("Error while creating application from url: %s", artifactUrlContent), e);
      }
    } else {
      throw new IllegalArgumentException(format("Invalid artifact type, only supported '%s' and '%s'", DOMAIN, APP));
    }
  }

  public void disposeIfNeeded(ToolingArtifact domainToolingArtifact) {
    if (domainToolingArtifact != null) {
      domainToolingArtifact.dispose();
    }
  }

  private Domain createDomain(String id, ArtifactResources artifactResources, Map<String, String> toolingArtifactProperties) {
    try {
      DomainDescriptor domainDescriptor =
          context.getDomainDescriptorFactory().create(artifactResources.getRootArtifactFile(), empty());
      DomainCache domainCache = context.getDomainCache();
      return domainCache
          .getDomain(id, () -> new DefaultDomain(id, artifactResources, domainDescriptor, context, toolingArtifactProperties));
    } catch (ToolingException e) {
      throw e;
    } catch (Exception e) {
      throw new ToolingException(format("Error while creating domain from url: %s", artifactResources.getArtifactUrlContent()),
                                 e);
    }
  }

  @Override
  public ToolingArtifact newToolingArtifact(URL artifactUrlContent, Map<String, String> toolingArtifactProperties,
                                            String parentId) {
    requireNonNull(artifactUrlContent, "artifactUrlContent cannot be null");
    requireNonNull(toolingArtifactProperties, "toolingArtifactProperties cannot be null");
    requireNonNull(parentId, "parentId cannot be null");

    Domain domain = context.getDomainCache().getDomain(parentId, () -> {
      throw new ToolingArtifactNotFoundException(
                                                 format("ToolingArtifact not found in Domain's cache for id: %s", parentId));
    });

    LazyValue<MetadataCache> parentMetadataCache = createMetadataCache(domain.getId(), domain);
    domain.setContext(context);
    ToolingArtifact domainToolingArtifact =
        newToolingArtifact(domain.getId(), context.getDomainCache(), parentMetadataCache, domain, context.getSerializer(),
                           context.getServiceRegistry());

    String id = randomUUID().toString();
    ArtifactResources artifactResources = new ArtifactResources(artifactUrlContent);

    MuleApplicationModel.MuleApplicationModelBuilder applicationArtifactModelBuilder =
        context.getApplicationDescriptorFactory().createArtifactModelBuilder(artifactResources.getRootArtifactFile());
    MuleArtifactLoaderDescriptor classLoaderModelDescriptorLoader =
        applicationArtifactModelBuilder.getClassLoaderModelDescriptorLoader();
    Map<String, Object> extendedAttributes = new HashMap<>(classLoaderModelDescriptorLoader.getAttributes());
    extendedAttributes.put(CLASSLOADER_MODEL_MAVEN_REACTOR_RESOLVER,
                           new DefaultToolingService.DomainMavenReactorResolver(
                                                                                domain
                                                                                    .getDescriptor().getArtifactLocation(),
                                                                                domain
                                                                                    .getDescriptor()
                                                                                    .getBundleDescriptor()));
    applicationArtifactModelBuilder
        .withClassLoaderModelDescriptorLoader(new MuleArtifactLoaderDescriptor(classLoaderModelDescriptorLoader.getId(),
                                                                               extendedAttributes));
    ApplicationDescriptor applicationDescriptor;
    try {
      applicationDescriptor = context.getApplicationDescriptorFactory()
          .createArtifact(artifactResources.getRootArtifactFile(), empty(), applicationArtifactModelBuilder.build());
    } catch (ToolingException e) {
      throw e;
    } catch (Exception e) {
      throw new ToolingException(format("Error while creating application from url: %s", artifactUrlContent), e);
    }

    checkDomain(domain.getDescriptor().getBundleDescriptor(), applicationDescriptor.getDomainDescriptor());
    applicationDescriptor.setDomainName(domain.getArtifactName());

    ApplicationCache applicationCache = context.getApplicationCache();
    Application application =
        applicationCache.getApplication(id, () -> new DefaultApplication(id, artifactResources, applicationDescriptor, domain,
                                                                         context, toolingArtifactProperties, true));
    LazyValue<MetadataCache> metadataCache = createMetadataCache(id, application);
    return newToolingArtifact(id, context.getApplicationCache(), metadataCache, application, domainToolingArtifact,
                              parentMetadataCache, context.getSerializer(), context.getServiceRegistry());
  }

  private void checkDomain(BundleDescriptor expectedDomainBundleDescriptor,
                           Optional<BundleDescriptor> applicationDeclaredDomainDescriptor) {
    if (!applicationDeclaredDomainDescriptor.isPresent()) {
      throw new ToolingException("Error while creating application", new IllegalStateException(
                                                                                               format("Application doesn't declared a domain dependency on its pom.xml to: '%s'",
                                                                                                      expectedDomainBundleDescriptor)));
    }
    BundleDescriptor actualBundleDescriptor = applicationDeclaredDomainDescriptor.get();
    if (!(expectedDomainBundleDescriptor.getGroupId().equals(actualBundleDescriptor.getGroupId())
        && expectedDomainBundleDescriptor.getArtifactId().equals(actualBundleDescriptor.getArtifactId())
        && expectedDomainBundleDescriptor.getVersion().equals(actualBundleDescriptor.getVersion()))) {
      throw new ToolingException("Error while creating application", new IllegalStateException(format(
                                                                                                      "Application declares a different domain dependency on its pom.xml expected: '%s' but was: '%s'",
                                                                                                      expectedDomainBundleDescriptor,
                                                                                                      actualBundleDescriptor)));
    }
  }

  private LazyValue<MetadataCache> createMetadataCache(String id, Artifact artifact) {
    return new LazyValue<>(() -> {
      MetadataCache metadataCache = NoOpMetadataCache.INSTANCE;
      if (artifact.getArtifactType().equals(APP)) {
        metadataCache = context.getMetadataCacheFactory()
            .map(factory -> factory.createMetadataCache(id, artifact.getArtifactUrlContent(), artifact.getProperties()))
            .orElse(NoOpMetadataCache.INSTANCE);
      }
      return metadataCache;
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException {
    Optional<Domain> domainOptional = context.getDomainCache().getDomain(id);
    if (domainOptional.isPresent()) {
      Domain domain = domainOptional.get();
      LazyValue<MetadataCache> domainMetadataCache = createMetadataCache(domain.getId(), domain);
      return createDomainToolingArtifact(domainOptional.get(), domainMetadataCache, context.getServiceRegistry());
    }
    Application application = context.getApplicationCache().getApplication(id, () -> {
      throw new ToolingArtifactNotFoundException(
                                                 format("ToolingArtifact not found in cache for id: %s", id));
    });
    LazyValue<MetadataCache> applicationMetadataCache = createMetadataCache(id, application);
    application.setContext(context);

    if (application.getDomain().isPresent()) {
      Domain domain = application.getDomain().get();
      LazyValue<MetadataCache> domainMetadataCache = createMetadataCache(domain.getId(), domain);
      ToolingArtifact domainToolingArtifact =
          createDomainToolingArtifact(domain, domainMetadataCache, context.getServiceRegistry());
      return newToolingArtifact(id, context.getApplicationCache(), applicationMetadataCache, application, domainToolingArtifact,
                                domainMetadataCache, context.getSerializer(), context.getServiceRegistry());
    }
    return newToolingArtifact(id, context.getApplicationCache(), applicationMetadataCache, application, null, null,
                              context.getSerializer(), context.getServiceRegistry());
  }

  private ToolingArtifact createDomainToolingArtifact(Domain domain, LazyValue<MetadataCache> domainMetadataCache,
                                                      ServiceRegistry serviceRegistry) {
    domain.setContext(context);
    return newToolingArtifact(domain.getId(), context.getDomainCache(), domainMetadataCache, domain, context.getSerializer(),
                              serviceRegistry);
  }

  private ToolingArtifact newToolingArtifact(final String id, final ApplicationCache applicationCache,
                                             final LazyValue<MetadataCache> metadataCache, final Application application,
                                             final ToolingArtifact parentToolingArtifact,
                                             final LazyValue<MetadataCache> parentMetadataCache,
                                             final Serializer serializer, final ServiceRegistry serviceRegistry) {
    return new ToolingArtifactWrapper(new DefaultToolingArtifact(id, metadataCache, application, parentToolingArtifact,
                                                                 parentMetadataCache, serializer, serviceRegistry) {

      @Override
      public void dispose() {
        applicationCache.invalidate(id);
        if (application.shouldDisposeDomain() && parentToolingArtifact != null) {
          parentToolingArtifact.dispose();
        }
      }
    });
  }

  private ToolingArtifact newToolingArtifact(final String id, final DomainCache domainCache,
                                             final LazyValue<MetadataCache> metadataCache, final Domain domain,
                                             final Serializer serializer, final ServiceRegistry serviceRegistry) {
    return new ToolingArtifactWrapper(new DefaultToolingArtifact(id, metadataCache, domain, serializer, serviceRegistry) {

      @Override
      public void dispose() {
        domainCache.invalidate(id);
      }
    });
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    Serializer serializer = context.getSerializer();
    switch (methodName) {
      case "messageHistoryService": {
        return this.messageHistoryService();
      }
      case "extensionModelService": {
        return this.extensionModelService();
      }
      case "artifactSerializationService": {
        return this.artifactSerializationService();
      }
      case "dslSyntaxResolverService": {
        return this.dslSyntaxResolverService();
      }
      case "iconsService": {
        return this.iconsService();
      }
      case "newToolingArtifact": {
        if (arguments.length == 3) {
          checkState(arguments.length == 3,
                     format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
          checkState(classes.length == 3 && classes[0].equals(URL.class.getName()),
                     format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
          checkState(classes.length == 3 && classes[1].equals(Map.class.getName()),
                     format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
          checkState(classes.length == 3 && classes[2].equals(String.class.getName()),
                     format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
          return newToolingArtifact(serializer.deserialize(arguments[0]), serializer.deserialize(arguments[1]),
                                    serializer.deserialize(arguments[2]));
        } else if (arguments.length == 2) {
          checkState(arguments.length == 2,
                     format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
          checkState(classes.length == 2 && classes[0].equals(URL.class.getName()),
                     format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
          checkState(classes.length == 2 && classes[1].equals(Map.class.getName()),
                     format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
          return newToolingArtifact(serializer.deserialize(arguments[0]), serializer.deserialize(arguments[1]));
        }
        checkState(arguments.length == 2 || arguments.length == 3,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
      }
      case "fetchToolingArtifact": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return fetchToolingArtifact(serializer.deserialize(arguments[0]));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
