/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;

import org.mule.tooling.client.api.feature.Feature;

import java.lang.reflect.Method;
import java.util.Arrays;

/**
 * Interface that allows to call this object by using reflection. This is needed in order to load dynamically
 * the implementation of Tooling and work in a separate class loader.
 *
 * @since 4.0
 */
public interface Command {

  /**
   * Dispatcher that calls the method on this object by the given name and arguments.
   * <p/>
   * Arguments are serialized to JSON.
   *
   * @param methodName the name of the method to be invoked.
   * @param classes array of class names for arguments.
   * @param arguments array of arguments serialized as JSON.
   * @return {@link Object}, in case of a service method invocation it will return the DTO serialized to JSON or the instance
   * in case of a service instance.
   */
  Object invokeMethod(String methodName, String[] classes, String[] arguments);

  default boolean isFeatureEnabled(String methodName, String[] classes) {
    final Class[] parameterTypes = Arrays.stream(classes).map(argClass -> {
      try {
        return this.getClass().getClassLoader().loadClass(argClass);
      } catch (ClassNotFoundException e) {
        throw new IllegalArgumentException(format("Feature '%s' is not supported with calling argument of type '%s'", methodName,
                                                  argClass));
      }
    }).collect(toList()).toArray(new Class[0]);

    try {
      Method method = this.getClass().getMethod(methodName, parameterTypes);
      if (!method.getReturnType().isAssignableFrom(Feature.class)) {
        throw new IllegalArgumentException(format("Method '%s' is not a Feature", methodName));
      }
      return true;
    } catch (NoSuchMethodException e) {
      return false;
    }
  }

  static RuntimeException methodNotFound(Class<? extends Command> clazz, String methodName) {
    return new IllegalArgumentException(format("Method '%s' not found on '%s'", methodName,
                                               clazz));
  }

  static RuntimeException notMatchingMethod(Class<? extends Command> clazz, String methodName, String[] classes) {
    return new IllegalArgumentException(format("Method '%s' not found on '%s' for parameter types '%s'", methodName,
                                               clazz, Arrays.toString(classes)));
  }

}
