/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Throwables.propagate;
import static com.google.common.base.Throwables.propagateIfPossible;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.concurrent.TimeUnit.SECONDS;
import org.mule.runtime.api.util.Preconditions;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.internal.application.Domain;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.util.concurrent.UncheckedExecutionException;

import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cache for {@link Domain}.
 */
public class DomainCache implements Disposable {

  private static final String TOOLING_CLIENT_DOMAIN_CACHE_EXPIRE_AFTER_ACCESS =
      "tooling.client.DomainCache.expireAfterAccess";
  private static final Logger LOGGER = LoggerFactory.getLogger(DomainCache.class);

  private Cache<String, Domain> domainsById = createCache();

  private Cache<String, Domain> createCache() {
    CacheBuilder builder = CacheBuilder.newBuilder();
    cacheExpireAfterAccess().ifPresent(value -> builder.expireAfterAccess(value, SECONDS));
    builder.removalListener((RemovalListener<String, Domain>) notification -> notification
        .getValue().dispose());
    return builder.build();
  }

  private Optional<Long> cacheExpireAfterAccess() {
    if (System.getProperty(TOOLING_CLIENT_DOMAIN_CACHE_EXPIRE_AFTER_ACCESS) == null) {
      return empty();
    }

    Long cacheSize = Long.valueOf(getProperty(TOOLING_CLIENT_DOMAIN_CACHE_EXPIRE_AFTER_ACCESS));
    Preconditions.checkArgument(cacheSize > 0,
                                format("Wrong value %d provided in system property %s, cacheExpireAfterAccess cannot be less that zero",
                                       cacheSize, TOOLING_CLIENT_DOMAIN_CACHE_EXPIRE_AFTER_ACCESS));
    return of(cacheSize);
  }

  public DomainCache() {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Initialising Domain cache");
    }
  }

  /**
   * Gets the {@link Domain} for the given id.
   *
   * @param id the id of the {@link Domain}to be retrieved.
   * @return the {@link Domain} or empty.
   */
  public Optional<Domain> getDomain(String id) {
    return ofNullable(domainsById.getIfPresent(id));
  }

  /**
   * Gets the {@link Domain} for the given id.
   * <p>
   * It uses a cache, so it will only call the {@code callable} if there's a cache miss.
   *
   * @param id the id of the {@link Domain}to be retrieved.
   * @param callable {@link Callable} to create a {@link Domain} in case of a cache miss.
   * @return the {@link Domain}
   */
  public Domain getDomain(String id,
                          Callable<Domain> callable) {
    try {
      return domainsById.get(id, callable);
    } catch (ExecutionException e) {
      if (e.getCause() instanceof RuntimeException) {
        throw (RuntimeException) e.getCause();
      } else {
        throw new RuntimeException(e.getCause());
      }
    } catch (UncheckedExecutionException e) {
      propagateIfPossible(e.getCause());
      throw propagate(e);
    }
  }

  public void invalidate(String id) {
    domainsById.invalidate(id);
  }

  @Override
  public void dispose() {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Disposing Domain cache");
    }
    this.domainsById.invalidateAll();
  }

}
