/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;


import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.success;
import static org.mule.tooling.client.internal.LocationFactory.toLocationDTO;

import org.mule.datasense.api.metadataprovider.CompatibleComponentAst;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataProvider;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.internal.datasense.DataSenseArtifact;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ToolingDataSenseMetadataProvider implements DataSenseMetadataProvider {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private MetadataResult failure = null;

  private DataSenseArtifact dataSenseArtifact;
  private long metadataTimeout;

  public ToolingDataSenseMetadataProvider(DataSenseArtifact dataSenseArtifact,
                                          long metadataTimeout) {
    this.dataSenseArtifact = dataSenseArtifact;
    this.metadataTimeout = metadataTimeout;
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(CompatibleComponentAst componentModel) {
    if (failure != null) {
      return failure;
    }
    final MetadataKeysRequest metadataKeysRequest = new MetadataKeysRequest();
    if (metadataTimeout > 0) {
      metadataKeysRequest.setRequestTimeout(metadataTimeout);
    }
    metadataKeysRequest.setLocation(toLocationDTO(componentModel.getLocation()));
    try {
      return dataSenseArtifact.getMetadataProvider(componentModel.getLocation()).getMetadataKeys(metadataKeysRequest);
    } catch (Exception e) {
      if (logger.isDebugEnabled()) {
        logger.debug("Error while resolving Metadata Keys for location: " + componentModel.getLocation(), e);
      }
      failure = failure(MetadataFailure.Builder.newFailure(e).onKeys());
      return failure;
    }
  }

  @Override
  public MetadataResult<OperationModel> getOperationMetadata(CompatibleComponentAst componentModel) {
    if (failure != null) {
      return failure;
    }
    try {
      MetadataResult<ComponentMetadataDescriptor<OperationModel>> operationMetadata =
          dataSenseArtifact.getMetadataProvider(componentModel.getLocation())
              .getOperationMetadata(getMetadataRequest(componentModel.getLocation()));
      if (operationMetadata.isSuccess()) {
        return success(operationMetadata.get().getModel());
      }
      return failure(operationMetadata.getFailures());
    } catch (Exception e) {
      if (logger.isDebugEnabled()) {
        logger.debug("Error while resolving Operation Metadata for location: " + componentModel.getLocation(), e);
      }
      failure = failure(MetadataFailure.Builder.newFailure(e).onComponent());
      return failure;
    }
  }

  @Override
  public MetadataResult<SourceModel> getSourceMetadata(CompatibleComponentAst componentModel) {
    if (failure != null) {
      return failure;
    }
    try {
      MetadataResult<ComponentMetadataDescriptor<SourceModel>> sourceMetadata =
          dataSenseArtifact.getMetadataProvider(componentModel.getLocation())
              .getSourceMetadata(getMetadataRequest(componentModel.getLocation()));
      if (sourceMetadata.isSuccess()) {
        return success(sourceMetadata.get().getModel());
      }
      return failure(sourceMetadata.getFailures());
    } catch (Exception e) {
      if (logger.isDebugEnabled()) {
        logger.debug("Error while resolving Source Metadata for location: " + componentModel.getLocation(), e);
      }
      failure = failure(MetadataFailure.Builder.newFailure(e).onComponent());
      return failure;
    }
  }

  private MetadataProvider.MetadataRequest getMetadataRequest(Location location) {
    MetadataProvider.MetadataRequest request = new MetadataProvider.MetadataRequest();
    request.setLocation(toLocationDTO(location));
    if (metadataTimeout > 0) {
      request.setRequestTimeout(metadataTimeout);
    }
    return request;
  }

}
