/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.dataweave;

import org.mule.runtime.api.metadata.MediaType;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;

import com.mulesoft.agent.domain.tooling.dataweave.model.Event;
import com.mulesoft.agent.domain.tooling.dataweave.model.Message;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewRequest;
import com.mulesoft.agent.domain.tooling.dataweave.model.PreviewResponse;
import com.mulesoft.agent.domain.tooling.dataweave.model.TypedValue;

import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;

/**
 * Base class for implementing {@link DataWeaveRunner}.
 *
 * @since 4.0
 */
public abstract class AbstractRemoteRunner implements DataWeaveRunner {

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    PreviewRequest request = toAgentPreviewRequest(parameters);
    PreviewResponse response = doExecute(request);
    return toDataWeavePreviewResponse(response);
  }

  protected abstract PreviewResponse doExecute(PreviewRequest request);

  private DataWeavePreviewResponse toDataWeavePreviewResponse(PreviewResponse response) {
    return DataWeavePreviewResponse.builder()
        .result(response.getResult())
        .mediaType(response.getMediaType())
        .valid(response.isValid())
        .errorMessage(response.getErrorMessage())
        .build();
  }

  private PreviewRequest toAgentPreviewRequest(DataWeavePreviewRequest parameters) {
    PreviewRequest request = new PreviewRequest();
    request.setScript(parameters.getScript());
    request.setTimeout(parameters.getRequestTimeout());

    EventModel parametersEvent = parameters.getEvent();
    if (parametersEvent != null) {
      Event event = new Event();
      toAgentMessage(parametersEvent, event);
      toAgentVariables(parametersEvent, event);
      request.setEvent(event);
    }

    return request;
  }

  private void toAgentVariables(EventModel parametersEvent, Event event) {
    Map<String, TypedValueModel> parametersVariables = parametersEvent.getVariables();
    if (parametersVariables != null && !parametersVariables.isEmpty()) {
      HashMap<String, TypedValue> variables = new HashMap<>();

      parametersVariables.forEach((s, typedValueModel) -> {
        if (typedValueModel != null) {
          variables.put(s, toAgentTypedValue(typedValueModel));
        }
      });

      event.setVariables(variables);
    }
  }

  private TypedValue toAgentTypedValue(TypedValueModel typedValueModel) {
    TypedValue value = new TypedValue();
    String mediaType = typedValueModel.getDataType().getMediaType();
    MediaType parse = MediaType.parse(mediaType);
    value.setValue(new String(typedValueModel.getContent(), parse.getCharset().orElseGet(Charset::defaultCharset)));
    value.setMediaType(mediaType);
    return value;
  }

  private void toAgentMessage(EventModel parametersEvent, Event event) {
    MessageModel parametersMessage = parametersEvent.getMessage();
    if (parametersMessage != null) {
      Message message = new Message();
      if (parametersMessage.getPayload() != null) {
        message.setPayload(toAgentTypedValue(parametersMessage.getPayload()));
      }
      if (parametersMessage.getAttributes() != null) {
        message.setAttributes(toAgentTypedValue(parametersMessage.getAttributes()));
      }
      event.setMessage(message);
    }
  }
}
