/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.log4j;

import static java.util.Objects.requireNonNull;
import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;

import java.net.URI;

import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.impl.Log4jContextFactory;
import org.apache.logging.log4j.core.selector.ClassLoaderContextSelector;
import org.apache.logging.log4j.core.selector.ContextSelector;
import org.apache.logging.log4j.core.util.Cancellable;
import org.apache.logging.log4j.core.util.ShutdownCallbackRegistry;

/**
 * Implementation of the {@link Log4jContextFactory} that reuses the same contextSelector and shutdownCallbackRegistry.
 * 
 * @since 4.0
 */
public class ToolingLog4jContextFactory extends Log4jContextFactory implements Disposable, ShutdownCallbackRegistry {

  private final ContextSelector selector;

  public ToolingLog4jContextFactory() {
    this(createToolingContextSelector(), createShutdownCallbackRegistry());
  }

  private ToolingLog4jContextFactory(final ContextSelector selector,
                                     final ShutdownCallbackRegistry shutdownCallbackRegistry) {
    super(selector, shutdownCallbackRegistry);
    this.selector = requireNonNull(selector, "No ContextSelector provided");
  }

  private static ContextSelector createToolingContextSelector() {
    return new ClassLoaderContextSelector() {

      @Override
      public LoggerContext getContext(String fqcn, ClassLoader loader, boolean currentContext) {
        if (loader instanceof ArtifactClassLoader) {
          return getDefault();
        }
        return super.getContext(fqcn, loader, currentContext);
      }

      @Override
      public LoggerContext getContext(String fqcn, ClassLoader loader, boolean currentContext, URI configLocation) {
        if (loader instanceof ArtifactClassLoader) {
          return getDefault();
        }
        return super.getContext(fqcn, loader, currentContext, configLocation);
      }
    };
  }

  private static ShutdownCallbackRegistry createShutdownCallbackRegistry() {
    return callback -> new Cancellable() {

      @Override
      public void cancel() {}

      @Override
      public void run() {}
    };
  }

  @Override
  public void dispose() {
    this.selector.getLoggerContexts().forEach(loggerContext -> {
      if (loggerContext != null && !loggerContext.isStopping() && !loggerContext.isStopped()) {
        loggerContext.stop();
      }
    });
  }

  @Override
  public Cancellable addShutdownCallback(Runnable callback) {
    return null;
  }

}
