/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.lang.String.format;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.deployment.model.api.domain.DomainDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.DefaultApplicationModelFactory;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.google.common.collect.ImmutableSet;

/**
 * Represents a Mule domain and allows to be deployed into Mule Runtime.
 *
 * @since 4.0
 */
public class DefaultDomain extends AbstractArtifact<DomainDescriptor> implements Domain {

  private ReentrantReadWriteLock deploymentLock = new ReentrantReadWriteLock();

  public DefaultDomain(String id, ArtifactResources artifactResources, DomainDescriptor domainDescriptor,
                       ToolingArtifactContext context,
                       Map<String, String> properties) {
    super(id, artifactResources, domainDescriptor, context, properties);
  }

  @Override
  public DomainDescriptor getDescriptor() {
    return artifactDescriptor;
  }

  @Override
  protected LazyValue<ToolingArtifactClassLoader> newToolingArtifactClassLoaderLazyValue() {
    return new LazyValue<>(() -> context.getDomainClassLoaderFactory().createDomainClassLoader(artifactDescriptor));
  }

  @Override
  protected LazyValue<ToolingApplicationModel> newToolingApplicationModelLazyValue() {
    return new LazyValue<>(() -> new ToolingApplicationModel(null,
                                                             new DefaultApplicationModelFactory()
                                                                 .createApplicationModel(artifactDescriptor,
                                                                                         ImmutableSet.<ExtensionModel>builder()
                                                                                             .addAll(getExtensionModels())
                                                                                             .build(),
                                                                                         getArtifactClassLoader()
                                                                                             .getClassLoader(),
                                                                                         noFailureConfigurationProperties,
                                                                                         getProperties())
                                                                 .orElseThrow(() -> new ToolingException(format("Couldn't create ApplicationModel from %s",
                                                                                                                this))),
                                                             getArtifactClassLoader().getClassLoader()));
  }


  @Override
  public <R> R evaluateWithRemoteDomain(DomainRemoteFunction<R> function) {
    checkState();
    if (!deployed) {
      Lock writeLock = deploymentLock.writeLock();
      if (writeLock.tryLock()) {
        try {
          if (!deployed) {
            deployed = true;
            remoteArtifactId = artifactDeployer.deploy(null);
            return function.apply(remoteArtifactId, runtimeToolingService.get());
          }
        } finally {
          writeLock.unlock();
        }
      }
    }

    Lock readLock = deploymentLock.readLock();
    readLock.lock();
    try {
      return function.apply(remoteArtifactId, runtimeToolingService.get());
    } finally {
      readLock.unlock();
    }
  }
}
