/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.lang.Long.parseLong;
import static java.lang.String.format;
import static java.util.Base64.getDecoder;
import static java.util.Base64.getEncoder;
import static java.util.Collections.emptyList;
import static org.mule.runtime.internal.dsl.DslConstants.CONFIG_ATTRIBUTE_NAME;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataFailuresDTO;
import static org.mule.tooling.client.internal.serialization.KryoFactory.defaultKryo;

import org.mule.datasense.api.metadataprovider.CompatibleComponentAst;
import org.mule.datasense.enrichment.model.IdComponentModelSelector;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.tooling.client.api.datasense.ImmutableMetadataCacheKeyInfo;
import org.mule.tooling.client.api.datasense.ImmutableMetadataResult;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataFailure;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import org.jetbrains.annotations.NotNull;

public class LegacyToolingMetadataCache implements MetadataCache {

  private final LazyValue<ApplicationModel> applicationModel;
  private final Map<String, String> toolingArtifactProperties;
  private final org.mule.tooling.client.api.datasense.MetadataCache delegate;
  private Kryo kryoSerializer;

  public LegacyToolingMetadataCache(LazyValue<ApplicationModel> applicationModel, Map<String, String> toolingArtifactProperties,
                                    org.mule.tooling.client.api.datasense.MetadataCache delegate) {
    this.applicationModel = applicationModel;
    this.toolingArtifactProperties = toolingArtifactProperties;
    this.delegate = delegate;
    this.kryoSerializer = defaultKryo();
  }

  @Override
  public MetadataResult<OperationModel> getOperationMetadata(CompatibleComponentAst componentAst,
                                                             Callable<MetadataResult<OperationModel>> resolver) {
    ComponentModel componentModel = getComponentModel(componentAst);
    return deserialize((String) delegate.getOperationMetadata(
                                                              new ImmutableMetadataCacheKeyInfo(getComponentId(componentModel),
                                                                                                componentAst.getLocation()
                                                                                                    .toString(),
                                                                                                resolveEffectiveTimestamp(componentModel),
                                                                                                toolingArtifactProperties),
                                                              () -> {
                                                                MetadataResult<OperationModel> result = resolver.call();
                                                                List<MetadataFailure> failures = emptyList();
                                                                if (!result.isSuccess()) {
                                                                  failures = toMetadataFailuresDTO(result.getFailures());
                                                                }
                                                                return new ImmutableMetadataResult(result.isSuccess(),
                                                                                                   serialize(result), failures);
                                                              }));
  }

  @Override
  public MetadataResult<SourceModel> getSourceMetadata(CompatibleComponentAst componentAst,
                                                       Callable<MetadataResult<SourceModel>> resolver) {
    ComponentModel componentModel = getComponentModel(componentAst);
    return deserialize((String) delegate.getSourceMetadata(
                                                           new ImmutableMetadataCacheKeyInfo(getComponentId(componentModel),
                                                                                             componentAst.getLocation()
                                                                                                 .toString(),
                                                                                             resolveEffectiveTimestamp(componentModel),
                                                                                             toolingArtifactProperties),
                                                           () -> {
                                                             MetadataResult<SourceModel> result = resolver.call();
                                                             List<MetadataFailure> failures = emptyList();
                                                             if (!result.isSuccess()) {
                                                               failures = toMetadataFailuresDTO(result.getFailures());
                                                             }
                                                             return new ImmutableMetadataResult(result.isSuccess(),
                                                                                                serialize(result), failures);
                                                           }));
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(CompatibleComponentAst componentAst,
                                                               Callable<MetadataResult<MetadataKeysContainer>> resolver) {
    ComponentModel componentModel = getComponentModel(componentAst);
    return deserialize((String) delegate.getMetadataKeys(
                                                         new ImmutableMetadataCacheKeyInfo(getComponentId(componentModel),
                                                                                           componentAst.getLocation().toString(),
                                                                                           resolveEffectiveTimestamp(componentModel),
                                                                                           toolingArtifactProperties),
                                                         () -> {
                                                           MetadataResult<MetadataKeysContainer> result = resolver.call();
                                                           List<MetadataFailure> failures = emptyList();
                                                           if (!result.isSuccess()) {
                                                             failures = toMetadataFailuresDTO(result.getFailures());
                                                           }
                                                           return new ImmutableMetadataResult(result.isSuccess(),
                                                                                              serialize(result), failures);
                                                         }));
  }

  @Override
  public void dispose(CompatibleComponentAst componentAst) {
    // Not supported
  }

  @Override
  public void invalidateMetadataKeysFor(CompatibleComponentAst componentAst) {
    // Not supported
  }

  @NotNull
  private ComponentModel getComponentModel(CompatibleComponentAst componentAst) {
    return componentAst.getModel(ComponentModel.class)
        .orElseThrow(() -> new IllegalArgumentException(format("A componentModel should be associated for %s",
                                                               componentAst.getLocation())));
  }

  private String serialize(Object object) {
    try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
      try (Output output = new Output(byteArrayOutputStream)) {
        kryoSerializer.writeClassAndObject(output, object);
      }
      return getEncoder().encodeToString(byteArrayOutputStream.toByteArray());
    } catch (IOException e) {
      throw new ToolingException("Error while creating object from serialization", e);
    }
  }

  private <T> T deserialize(String content) {
    try (Input input = new Input(new ByteArrayInputStream(getDecoder().decode(content)))) {
      return (T) kryoSerializer.readClassAndObject(input);
    }
  }

  private Optional<Long> resolveTimestamp(ComponentModel componentModel) {
    if (componentModel == null) {
      return Optional.empty();
    }
    try {
      return Optional.of(parseLong(componentModel.getParameters().get("doc:timestamp")));
    } catch (NumberFormatException e) {
      return Optional.empty();
    }
  }

  private Optional<String> getConfigurationRef(ComponentModel componentModel) {
    return Optional.ofNullable(componentModel.getParameters().get(CONFIG_ATTRIBUTE_NAME));
  }

  private String getComponentId(ComponentModel componentModel) {
    return IdComponentModelSelector.getComponentId(componentModel);
  }

  private Long resolveEffectiveTimestamp(ComponentModel componentModel) {
    final Long componentTimestamp = resolveTimestamp(componentModel).orElse(null);
    final Optional<ComponentModel> relatedConfigurationOptional =
        getConfigurationRef(componentModel).flatMap(applicationModel.get()::findTopLevelNamedComponent);
    if (relatedConfigurationOptional.isPresent()) {
      final Long configurationTimestamp = resolveTimestamp(relatedConfigurationOptional.get()).orElse(null);
      return componentTimestamp != null && configurationTimestamp != null ? Long.max(componentTimestamp, configurationTimestamp)
          : null;
    } else {
      return componentTimestamp;
    }
  }


}
