/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static org.slf4j.LoggerFactory.getLogger;
import org.mule.datasense.api.metadataprovider.CompatibleComponentAst;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.config.api.dsl.model.metadata.ModelBasedMetadataCacheIdGeneratorFactory;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheId;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGenerator;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGeneratorFactory;
import org.mule.runtime.dsl.api.component.config.ComponentConfiguration;

import com.vdurmont.semver4j.Semver;
import com.vdurmont.semver4j.SemverException;

import java.util.Optional;
import java.util.function.Supplier;

import org.slf4j.Logger;

/**
 * Generates keys to use in a {@link MetadataCache}
 *
 * @since 4.2.2, 4.3.0
 */
public class ToolingMetadataCacheIdGenerator {

  private static final Logger LOGGER = getLogger(ToolingMetadataCacheIdGenerator.class);
  private static final String ENRICHED_PART_SEPARATOR = "#";

  private static final ModelBasedMetadataCacheIdGeneratorFactory cacheIdGeneratorFactory =
      new ModelBasedMetadataCacheIdGeneratorFactory();

  private MetadataCacheIdGenerator<ComponentConfiguration> delegate;
  private DslResolvingContext dslResolvingContext;

  /**
   * Returns whether or not the {@link String}s represent keys that are related.
   * That is true if the @param child has a reference to it's parent.
   * </p>
   * For example, if an operation (child) has a reference to a config (parent)
   *
   * @param parentValue
   * @param childValue
   * @return {@link Boolean#TRUE} if they are related, {@link Boolean#FALSE} otherwise.
   */
  public static boolean areRelated(String parentValue, String childValue) {
    Pair<String, String> parentSplitInfo = splitInformation(parentValue); //EnrichedValue, DefaultValue
    Pair<String, String> childSplitInfo = splitInformation(childValue); //EnrichedValue, DefaultValue
    if (parentSplitInfo.getFirst().equals(childSplitInfo.getFirst())) {
      return childSplitInfo.getSecond().contains(parentSplitInfo.getSecond());
    }
    return false;
  }

  private static Pair<String, String> splitInformation(String idValue) {
    String enrichedPart = "";
    String valuePart;
    String[] parts = idValue.split(ENRICHED_PART_SEPARATOR);
    if (parts.length == 1) {
      valuePart = parts[0];
    } else {
      enrichedPart = parts[0];
      valuePart = parts[1];
    }
    return new Pair<>(enrichedPart, valuePart);
  }

  private String betweenBrackets(String value) {
    return "[" + value + "]";
  }

  public ToolingMetadataCacheIdGenerator(DslResolvingContext context,
                                         MetadataCacheIdGeneratorFactory.ComponentLocator<ComponentConfiguration> locator) {
    this.dslResolvingContext = context;
    this.delegate = cacheIdGeneratorFactory.create(context, locator);
  }

  public Optional<String> getIdForComponentOutputMetadata(CompatibleComponentAst component) {
    return withConfigurationModel(component)
        .flatMap(cm -> generateMetadataCacheId(component, () -> delegate.getIdForComponentOutputMetadata(cm)));
  }

  public Optional<String> getIdForComponentAttributesMetadata(CompatibleComponentAst component) {
    return withConfigurationModel(component)
        .flatMap(cm -> generateMetadataCacheId(component, () -> delegate.getIdForComponentAttributesMetadata(cm)));
  }

  public Optional<String> getIdForComponentInputMetadata(CompatibleComponentAst component, String parameterName) {
    return withConfigurationModel(component)
        .flatMap(cm -> generateMetadataCacheId(component, () -> delegate.getIdForComponentInputMetadata(cm, parameterName)));
  }

  public Optional<String> getIdForComponentMetadata(CompatibleComponentAst component) {
    return withConfigurationModel(component)
        .flatMap(cm -> generateMetadataCacheId(component, () -> delegate.getIdForComponentMetadata(cm)));
  }

  public Optional<String> getIdForMetadataKeys(CompatibleComponentAst component) {
    return withConfigurationModel(component)
        .flatMap(cm -> generateMetadataCacheId(component, () -> delegate.getIdForMetadataKeys(cm)));
  }

  public Optional<String> getIdForGlobalMetadata(CompatibleComponentAst component) {
    return withConfigurationModel(component)
        .flatMap(cm -> generateMetadataCacheId(component, () -> delegate.getIdForGlobalMetadata(cm)));
  }

  private Optional<String> generateMetadataCacheId(CompatibleComponentAst componentAst,
                                                   Supplier<Optional<MetadataCacheId>> metadataCacheIdSupplier) {
    return metadataCacheIdSupplier.get().map(
                                             metadataCacheId -> {
                                               Reference<String> idToReturn = new Reference<>(metadataCacheId.getValue());
                                               getExtensionModel(componentAst).ifPresent(em -> {
                                                 try {
                                                   String major = getMajor(em.getVersion());
                                                   idToReturn.set(betweenBrackets(em.getName().toLowerCase()) +
                                                       "-" +
                                                       betweenBrackets(major) +
                                                       ENRICHED_PART_SEPARATOR +
                                                       metadataCacheId.getValue());
                                                 } catch (SemverException e) {
                                                   if (LOGGER.isErrorEnabled()) {
                                                     LOGGER.error("Unable to properly create a cache key id for " + componentAst
                                                         + " ,extension model version does not follow semver, was: "
                                                         + em.getVersion());
                                                   }
                                                 }
                                               });
                                               return idToReturn.get();
                                             });
  }

  private Optional<ExtensionModel> getExtensionModel(CompatibleComponentAst componentAst) {
    return componentAst.getModel(ComponentModel.class).flatMap(
                                                               cm -> dslResolvingContext
                                                                   .getExtensions().stream().filter(em -> cm.getIdentifier()
                                                                       .getNamespace().equals(em.getXmlDslModel().getPrefix()))
                                                                   .findAny());
  }

  private String getMajor(String version) {
    Semver semver = new Semver(version);
    return semver.getMajor().toString();
  }

  private Optional<ComponentConfiguration> withConfigurationModel(CompatibleComponentAst configurationModel) {
    return configurationModel.getModel(ComponentModel.class).map(ComponentModel::getConfiguration);
  }
}
