/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toClassValueModelDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toExpressionSupportDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toLayoutModelDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toPathModelDTO;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.toStereotypesDTO;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.meta.model.display.ClassValueModel;
import org.mule.runtime.api.meta.model.display.LayoutModel;
import org.mule.runtime.api.meta.model.display.PathModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.tooling.client.api.declaration.type.annotation.DefaultImplementingTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.DisplayTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.DslBaseType;
import org.mule.tooling.client.api.declaration.type.annotation.ExclusiveOptionalsTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.ExpressionSupportAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.ExtensibleTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.FlattenedTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.LayoutTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.ParameterDslAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.QNameTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.StereotypeTypeAnnotation;
import org.mule.tooling.client.api.declaration.type.annotation.SubstitutionGroup;
import org.mule.tooling.client.api.declaration.type.annotation.TypeDslAnnotation;
import org.mule.tooling.client.api.extension.model.DisplayModel;

import com.google.common.collect.ImmutableMap;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.collections4.Transformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TypeAnnotationsTransformers {

  private static Logger LOGGER = LoggerFactory.getLogger(TypeAnnotationsTransformers.class);

  public static Map<Class, Transformer<TypeAnnotation, TypeAnnotation>> typeAnnotationTransformers =
      ImmutableMap.<Class, Transformer<TypeAnnotation, TypeAnnotation>>builder()
          // extensions api
          .put(org.mule.runtime.extension.api.declaration.type.annotation.DisplayTypeAnnotation.class,
               typeAnnotation -> {
                 org.mule.runtime.extension.api.declaration.type.annotation.DisplayTypeAnnotation displayTypeAnnotation =
                     (org.mule.runtime.extension.api.declaration.type.annotation.DisplayTypeAnnotation) typeAnnotation;
                 DisplayModel displayModel =
                     new DisplayModel(displayTypeAnnotation.getDisplayName(), displayTypeAnnotation.getSummary(),
                                      displayTypeAnnotation.getExample(), toPathModelDTO(displayTypeAnnotation.getPathModel()),
                                      toClassValueModelDTO(displayTypeAnnotation.getClassValueModel()));
                 return new DisplayTypeAnnotation(displayModel);
               })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.TypeDslAnnotation.class,
               typeAnnotation -> {
                 org.mule.runtime.extension.api.declaration.type.annotation.TypeDslAnnotation typeDslAnnotation =
                     (org.mule.runtime.extension.api.declaration.type.annotation.TypeDslAnnotation) typeAnnotation;

                 Optional<SubstitutionGroup> substitutionGroup = empty();
                 if (typeDslAnnotation.getSubstitutionGroup().isPresent()) {
                   org.mule.runtime.extension.api.declaration.type.annotation.SubstitutionGroup originalSubstitutionGroup =
                       typeDslAnnotation.getSubstitutionGroup().get();
                   substitutionGroup =
                       of(new SubstitutionGroup(originalSubstitutionGroup.getPrefix(), originalSubstitutionGroup.getElement()));
                 }

                 Optional<DslBaseType> dslBaseType = empty();
                 if (typeDslAnnotation.getDslBaseType().isPresent()) {
                   org.mule.runtime.extension.api.declaration.type.annotation.DslBaseType originalDslBaseType =
                       typeDslAnnotation.getDslBaseType().get();
                   dslBaseType = of(new DslBaseType(originalDslBaseType.getPrefix(), originalDslBaseType.getType()));
                 }


                 return new TypeDslAnnotation(typeDslAnnotation.allowsInlineDefinition(),
                                              typeDslAnnotation.allowsTopLevelDefinition(),
                                              substitutionGroup.orElse(null), dslBaseType.orElse(null));
               })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.ExpressionSupportAnnotation.class,
               typeAnnotation -> {
                 org.mule.runtime.extension.api.declaration.type.annotation.ExpressionSupportAnnotation expressionSupportAnnotation =
                     (org.mule.runtime.extension.api.declaration.type.annotation.ExpressionSupportAnnotation) typeAnnotation;
                 return new ExpressionSupportAnnotation(toExpressionSupportDTO(expressionSupportAnnotation
                     .getExpressionSupport()));
               })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.ExclusiveOptionalsTypeAnnotation.class,
               typeAnnotation -> {
                 org.mule.runtime.extension.api.declaration.type.annotation.ExclusiveOptionalsTypeAnnotation exclusiveOptionalsTypeAnnotation =
                     (org.mule.runtime.extension.api.declaration.type.annotation.ExclusiveOptionalsTypeAnnotation) typeAnnotation;
                 return new ExclusiveOptionalsTypeAnnotation(exclusiveOptionalsTypeAnnotation.getExclusiveParameterNames(),
                                                             exclusiveOptionalsTypeAnnotation.isOneRequired());
               })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.ExtensibleTypeAnnotation.class,
               typeAnnotation -> new org.mule.tooling.client.api.declaration.type.annotation.ExtensibleTypeAnnotation())
          .put(org.mule.runtime.extension.api.declaration.type.annotation.FlattenedTypeAnnotation.class,
               typeAnnotation -> new org.mule.tooling.client.api.declaration.type.annotation.FlattenedTypeAnnotation())
          .put(org.mule.runtime.extension.api.declaration.type.annotation.LayoutTypeAnnotation.class, typeAnnotation -> {
            org.mule.runtime.extension.api.declaration.type.annotation.LayoutTypeAnnotation layoutTypeAnnotation =
                (org.mule.runtime.extension.api.declaration.type.annotation.LayoutTypeAnnotation) typeAnnotation;
            return new LayoutTypeAnnotation(toLayoutModelDTO(of(layoutTypeAnnotation.getLayoutModel())));
          })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.ParameterDslAnnotation.class, typeAnnotation -> {
            org.mule.runtime.extension.api.declaration.type.annotation.ParameterDslAnnotation parameterDslAnnotation =
                (org.mule.runtime.extension.api.declaration.type.annotation.ParameterDslAnnotation) typeAnnotation;
            return new ParameterDslAnnotation(parameterDslAnnotation.allowsInlineDefinition(),
                                              parameterDslAnnotation.allowsReferences());
          })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.QNameTypeAnnotation.class, typeAnnotation -> {
            org.mule.runtime.extension.api.declaration.type.annotation.QNameTypeAnnotation qNameTypeAnnotation =
                (org.mule.runtime.extension.api.declaration.type.annotation.QNameTypeAnnotation) typeAnnotation;
            return new QNameTypeAnnotation(qNameTypeAnnotation.getValue());
          })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation.class, typeAnnotation -> {
            org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation stereotypeTypeAnnotation =
                (org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation) typeAnnotation;
            return new StereotypeTypeAnnotation(toStereotypesDTO(stereotypeTypeAnnotation.getAllowedStereotypes(),
                                                                 new ArrayList<>()));
          })
          .put(org.mule.runtime.extension.api.declaration.type.annotation.DefaultImplementingTypeAnnotation.class,
               typeAnnotation -> {
                 org.mule.runtime.extension.api.declaration.type.annotation.DefaultImplementingTypeAnnotation defaultImplementingTypeAnnotation =
                     (org.mule.runtime.extension.api.declaration.type.annotation.DefaultImplementingTypeAnnotation) typeAnnotation;
                 return new DefaultImplementingTypeAnnotation(defaultImplementingTypeAnnotation.getDefaultType());
               })
          .build();

  public static Map<Class, Transformer<TypeAnnotation, TypeAnnotation>> toolingTypeAnnotationTransformers =
      ImmutableMap.<Class, Transformer<TypeAnnotation, TypeAnnotation>>builder()
          // extensions api
          .put(DisplayTypeAnnotation.class,
               toolingTypeAnnotation -> {
                 DisplayTypeAnnotation displayTypeAnnotation = (DisplayTypeAnnotation) toolingTypeAnnotation;
                 final org.mule.runtime.api.meta.model.display.DisplayModel.DisplayModelBuilder displayModelBuilder =
                     org.mule.runtime.api.meta.model.display.DisplayModel.builder()
                         .displayName(displayTypeAnnotation.getDisplayName())
                         .summary(displayTypeAnnotation.getSummary())
                         .example(displayTypeAnnotation.getExample());
                 displayTypeAnnotation.getPathModel().ifPresent(pathModel -> {
                   pathModel.getFileExtensions();
                   PathModel.Type type = PathModel.Type.ANY;
                   try {
                     type = PathModel.Type.valueOf(pathModel.getType().getValue());
                   } catch (IllegalArgumentException e) {
                     if (LOGGER.isWarnEnabled()) {
                       LOGGER.warn("Couldn't find a PathModel.Type for name: {}", pathModel.getType().getValue());
                     }
                   }
                   displayModelBuilder
                       .path(new PathModel(type, pathModel.acceptsUrls(),
                                           PathModel.Location.valueOf(pathModel.getLocation().getValue()),
                                           pathModel.getFileExtensions().toArray(new String[] {})));
                 });
                 displayTypeAnnotation.getClassValueModel()
                     .ifEnabled(classValueModelOptional -> classValueModelOptional
                         .ifPresent(classValueModel -> displayModelBuilder
                             .classValue(new ClassValueModel(classValueModel.getAssignableFrom()))));

                 return new org.mule.runtime.extension.api.declaration.type.annotation.DisplayTypeAnnotation(displayModelBuilder
                     .build());
               })
          .put(TypeDslAnnotation.class,
               toolingTypeAnnotation -> {
                 TypeDslAnnotation typeDslAnnotation = (TypeDslAnnotation) toolingTypeAnnotation;
                 return new org.mule.runtime.extension.api.declaration.type.annotation.TypeDslAnnotation(
                                                                                                         typeDslAnnotation
                                                                                                             .allowsInlineDefinition(),
                                                                                                         typeDslAnnotation
                                                                                                             .allowsTopLevelDefinition(),
                                                                                                         typeDslAnnotation
                                                                                                             .getSubstitutionGroup()
                                                                                                             .map(substitutionGroup -> substitutionGroup
                                                                                                                 .getFormattedAnnotation())
                                                                                                             .orElse(null),
                                                                                                         typeDslAnnotation
                                                                                                             .getDslBaseType()
                                                                                                             .map(dslBaseType -> dslBaseType
                                                                                                                 .getFormattedAnnotation())
                                                                                                             .orElse(null));
               })
          .put(ExpressionSupportAnnotation.class,
               toolingTypeAnnotation -> {
                 ExpressionSupportAnnotation expressionSupportAnnotation = (ExpressionSupportAnnotation) toolingTypeAnnotation;
                 ExpressionSupport expressionSupport = ExpressionSupport.NOT_SUPPORTED;
                 try {
                   expressionSupport = ExpressionSupport.valueOf(expressionSupportAnnotation.getExpressionSupport().getValue());
                 } catch (IllegalArgumentException e) {
                   if (LOGGER.isWarnEnabled()) {
                     LOGGER.warn("Couldn't find a ExpressionSupport for name: {}",
                                 expressionSupportAnnotation.getExpressionSupport().getValue());
                   }
                 }
                 return new org.mule.runtime.extension.api.declaration.type.annotation.ExpressionSupportAnnotation(expressionSupport);
               })
          .put(ExclusiveOptionalsTypeAnnotation.class,
               toolingTypeAnnotation -> {
                 ExclusiveOptionalsTypeAnnotation exclusiveOptionalsTypeAnnotation =
                     (ExclusiveOptionalsTypeAnnotation) toolingTypeAnnotation;
                 return new org.mule.runtime.extension.api.declaration.type.annotation.ExclusiveOptionalsTypeAnnotation(exclusiveOptionalsTypeAnnotation
                     .getExclusiveParameterNames(),
                                                                                                                        exclusiveOptionalsTypeAnnotation
                                                                                                                            .isOneRequired());
               })
          .put(ExtensibleTypeAnnotation.class,
               toolingTypeAnnotation -> new org.mule.runtime.extension.api.declaration.type.annotation.ExtensibleTypeAnnotation())
          .put(FlattenedTypeAnnotation.class,
               toolingTypeAnnotation -> new org.mule.runtime.extension.api.declaration.type.annotation.FlattenedTypeAnnotation())
          .put(LayoutTypeAnnotation.class, toolingTypeAnnotation -> {
            LayoutTypeAnnotation layoutTypeAnnotation = (LayoutTypeAnnotation) toolingTypeAnnotation;
            LayoutModel.LayoutModelBuilder layoutModelBuilder = LayoutModel.builder();
            if (layoutTypeAnnotation.getLayoutModel().isPassword()) {
              layoutModelBuilder.asPassword();
            }
            if (layoutTypeAnnotation.getLayoutModel().isQuery()) {
              layoutModelBuilder.asQuery();
            }
            if (layoutTypeAnnotation.getLayoutModel().isText()) {
              layoutModelBuilder.asText();
            }

            layoutTypeAnnotation.getLayoutModel().getOrder().ifPresent(order -> layoutModelBuilder.order(order));
            layoutTypeAnnotation.getLayoutModel().getTabName().ifPresent(tabName -> layoutModelBuilder.tabName(tabName));
            return new org.mule.runtime.extension.api.declaration.type.annotation.LayoutTypeAnnotation(layoutModelBuilder
                .build());
          })
          .put(ParameterDslAnnotation.class, toolingTypeAnnotation -> {
            ParameterDslAnnotation parameterDslAnnotation = (ParameterDslAnnotation) toolingTypeAnnotation;
            return new org.mule.runtime.extension.api.declaration.type.annotation.ParameterDslAnnotation(parameterDslAnnotation
                .allowsInlineDefinition(),
                                                                                                         parameterDslAnnotation
                                                                                                             .allowsReferences());
          })
          .put(QNameTypeAnnotation.class, toolingTypeAnnotation -> {
            QNameTypeAnnotation qNameTypeAnnotation = (QNameTypeAnnotation) toolingTypeAnnotation;
            return new org.mule.runtime.extension.api.declaration.type.annotation.QNameTypeAnnotation(qNameTypeAnnotation
                .getValue());
          })
          .put(StereotypeTypeAnnotation.class, toolingTypeAnnotation -> {
            StereotypeTypeAnnotation stereotypeTypeAnnotation = (StereotypeTypeAnnotation) toolingTypeAnnotation;
            List<StereotypeModel> allowedStereotypes = stereotypeTypeAnnotation
                .getStereotypeModel().stream().map(stereotypeModel -> toStereotypeModel(stereotypeModel)).collect(toList());
            return new org.mule.runtime.extension.api.declaration.type.annotation.StereotypeTypeAnnotation(allowedStereotypes);
          })
          .put(DefaultImplementingTypeAnnotation.class,
               toolingTypeAnnotation -> {
                 DefaultImplementingTypeAnnotation defaultImplementingTypeAnnotation =
                     (DefaultImplementingTypeAnnotation) toolingTypeAnnotation;
                 return new org.mule.runtime.extension.api.declaration.type.annotation.DefaultImplementingTypeAnnotation(defaultImplementingTypeAnnotation
                     .getDefaultType());
               })
          .build();

  private static org.mule.runtime.api.meta.model.stereotype.StereotypeModel toStereotypeModel(
                                                                                              org.mule.tooling.client.api.extension.model.StereotypeModel stereotypeModel) {
    return new org.mule.runtime.api.meta.model.stereotype.ImmutableStereotypeModel(stereotypeModel.getType(),
                                                                                   stereotypeModel.getNamespace(),
                                                                                   stereotypeModel.getParent()
                                                                                       .map(parent -> toStereotypeModel(parent))
                                                                                       .orElse(null));
  }


}
