/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.util.Optional.of;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.sameInstance;
import static org.hamcrest.core.IsEqual.equalTo;
import static org.junit.Assert.assertThat;
import static org.junit.Assert.fail;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.success;

import org.mule.datasense.api.metadataprovider.CompatibleComponentAst;
import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.OutputModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.MetadataKeysContainerBuilder;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.extension.api.metadata.NullMetadataKey;
import org.mule.runtime.extension.api.model.ImmutableOutputModel;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class DefaultToolingMetadataCacheTestCase extends AbstractMetadataTestCase {

  private Map<String, MetadataType> metadataTypesCache;
  private Map<String, List<MetadataFailure>> failureMetadataComponentCache;
  private Map<String, MetadataResult<MetadataKeysContainer>> metadataKeysCache;

  private ToolingMetadataCacheIdGenerator metadataCacheIdGenerator;

  private MetadataCache metadataCache;

  @Before
  public void prepare() {
    metadataTypesCache = new HashMap<>();
    failureMetadataComponentCache = new HashMap<>();
    metadataKeysCache = new HashMap<>();

    metadataCacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    metadataCache = new DefaultToolingMetadataCache(metadataTypesCache,
                                                    failureMetadataComponentCache,
                                                    metadataKeysCache,
                                                    metadataCacheIdGenerator);


  }

  @Test
  public void failureResolvingMetadataKeys() {
    CompatibleComponentAst componentAst = mock(CompatibleComponentAst.class);

    final String keysKey = "keys";
    when(metadataCacheIdGenerator.getIdForMetadataKeys(componentAst)).thenReturn(of(keysKey));

    MetadataResult failure = failure("Invalid model", newFailure().withMessage("Mocked Runtime").onKeys());

    MetadataResult<MetadataKeysContainer> metadataResult = metadataCache.getMetadataKeys(componentAst, () -> failure);

    assertThat(metadataResult.isSuccess(), is(false));

    assertThat(metadataTypesCache.size(), is(0));
    assertThat(failureMetadataComponentCache.size(), is(0));

    assertThat(metadataKeysCache.size(), is(1));
    assertThat(metadataKeysCache.containsKey(keysKey), is(true));
    assertThat(metadataKeysCache.get(keysKey).getFailures(), equalTo(failure.getFailures()));
  }

  @Test
  public void resolveMetadataKeys() {
    CompatibleComponentAst componentAst = mock(CompatibleComponentAst.class);

    final String keysKey = "keys";
    when(metadataCacheIdGenerator.getIdForMetadataKeys(componentAst)).thenReturn(of(keysKey));

    MetadataResult success =
        success(MetadataKeysContainerBuilder.getInstance().add("key", ImmutableSet.of(new NullMetadataKey())).build());

    MetadataResult<MetadataKeysContainer> metadataResult = metadataCache.getMetadataKeys(componentAst, () -> success);

    assertThat(metadataResult.isSuccess(), is(true));

    assertThat(metadataTypesCache.size(), is(0));
    assertThat(failureMetadataComponentCache.size(), is(0));

    assertThat(metadataKeysCache.size(), is(1));
    assertThat(metadataKeysCache.containsKey(keysKey), is(true));
    assertThat(metadataKeysCache.get(keysKey).isSuccess(), is(true));
    assertThat(metadataKeysCache.get(keysKey), sameInstance(success));
    assertThat(metadataKeysCache.get(keysKey).getFailures().size(), is(0));
  }

  @Test
  public void failureResolvingMetadataShouldBeAddedToCacheAsComponentId() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    CompatibleComponentAst componentAst = mock(CompatibleComponentAst.class);
    when(componentAst.getModel(eq(OperationModel.class))).thenReturn(of(operationModel));

    final String outputKey = "output";
    when(metadataCacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(outputKey));

    final String componentKey = "component";
    when(metadataCacheIdGenerator.getIdForComponentMetadata(componentAst)).thenReturn(of(componentKey));

    MetadataResult failure = failure("Invalid model", newFailure().withMessage("Mocked Runtime").onComponent());

    MetadataResult<OperationModel> metadataResult = metadataCache.getOperationMetadata(componentAst, () -> failure);

    assertThat(metadataResult.isSuccess(), is(false));

    assertThat(metadataTypesCache.size(), is(0));
    assertThat(metadataTypesCache.containsKey(outputKey), is(false));

    assertThat(metadataKeysCache.size(), is(0));

    assertThat(failureMetadataComponentCache.size(), is(1));
    assertThat(failureMetadataComponentCache.containsKey(componentKey), is(true));
    assertThat(failureMetadataComponentCache.get(componentKey), equalTo(failure.getFailures()));
  }

  @Test
  public void failureByComponentIdShouldBeFetchedFromCache() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    CompatibleComponentAst componentAst = mock(CompatibleComponentAst.class);
    when(componentAst.getModel(eq(OperationModel.class))).thenReturn(of(operationModel));

    final String componentKey = "component";
    when(metadataCacheIdGenerator.getIdForComponentMetadata(componentAst)).thenReturn(of(componentKey));

    MetadataResult failure = failure("Invalid model", newFailure().withMessage("Mocked Runtime").onComponent());

    failureMetadataComponentCache.put(componentKey, failure.getFailures());

    MetadataResult<OperationModel> metadataResult = metadataCache.getOperationMetadata(componentAst, () -> {
      fail("Should not be called to resolve metadata");
      return null;
    });

    assertThat(metadataResult.isSuccess(), is(false));
    assertThat(metadataResult.getFailures(), equalTo(failure.getFailures()));
    assertThat(failureMetadataComponentCache.size(), is(1));
    assertThat(failureMetadataComponentCache.containsKey(componentKey), is(true));
    assertThat(failureMetadataComponentCache.get(componentKey), equalTo(failure.getFailures()));

    assertThat(metadataTypesCache.size(), is(0));
    assertThat(metadataKeysCache.size(), is(0));
  }

  @Test
  public void successResolveMetadataForOutputModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(dynamicParameterModel), dynamicOutputModel, dynamicOutputModel);

    CompatibleComponentAst componentAst = mock(CompatibleComponentAst.class);
    when(componentAst.getModel(eq(OperationModel.class))).thenReturn(of(operationModel));

    final String componentKey = "component";
    when(metadataCacheIdGenerator.getIdForComponentMetadata(componentAst)).thenReturn(of(componentKey));

    final String parameterKey = "parameter";
    when(metadataCacheIdGenerator.getIdForComponentInputMetadata(componentAst, dynamicParameterModel.getName()))
        .thenReturn(of(parameterKey));
    metadataTypesCache.put(parameterKey, objectType);

    final String outputKey = "output";
    when(metadataCacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(outputKey));

    final String attributesKey = "attributes";
    when(metadataCacheIdGenerator.getIdForComponentAttributesMetadata(componentAst))
        .thenReturn(of(attributesKey));
    metadataTypesCache.put(attributesKey, objectType);

    MetadataResult<OperationModel> metadataResult = metadataCache.getOperationMetadata(componentAst, () -> {
      OutputModel resolvedOutputModel = new ImmutableOutputModel(DYNAMIC_OUTPUT_MODEL_DESCRIPTION,
                                                                 objectType,
                                                                 true,
                                                                 modelProperties);

      return success(getOperationModel(ImmutableList.of(dynamicParameterModel), resolvedOutputModel, dynamicOutputModel));
    });

    assertThat(metadataResult.isSuccess(), is(true));
    assertThat(metadataTypesCache.size(), is(3));

    assertThat(metadataTypesCache.get(parameterKey), sameInstance(objectType));
    assertThat(metadataTypesCache.get(attributesKey), sameInstance(objectType));
    assertThat(metadataTypesCache.get(outputKey), sameInstance(objectType));

    assertThat(failureMetadataComponentCache.size(), is(0));
    assertThat(metadataKeysCache.size(), is(0));
  }

}
