/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static java.util.Objects.requireNonNull;
import static org.codehaus.plexus.util.ExceptionUtils.getStackTrace;
import static org.mule.runtime.api.component.location.Location.builderFromStringRepresentation;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.runtime.api.value.ResolvingFailure.Builder.newFailure;
import static org.mule.runtime.api.value.ValueResult.resultFrom;
import static org.mule.runtime.extension.api.values.ValueResolvingException.INVALID_LOCATION;
import static org.mule.runtime.extension.api.values.ValueResolvingException.UNKNOWN;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.core.internal.locator.ComponentLocator;
import org.mule.runtime.deployment.model.api.DeploymentException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.value.ResolvingFailure;
import org.mule.tooling.client.api.value.Value;
import org.mule.tooling.client.api.value.ValueResult;
import org.mule.tooling.client.api.value.provider.ValueProviderDisposeRequest;
import org.mule.tooling.client.api.value.provider.ValueProviderRequest;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.action.DispacheableAction;
import org.mule.tooling.client.internal.cache.CacheWrappedException;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.values.ValueProviderCache;

import java.util.HashSet;
import java.util.Optional;
import java.util.concurrent.CompletionException;

/**
 * Provides the capability of resolve the {@link Value values} for any Value Provider capable element in a Mule Domain or Application.
 *
 * @since 1.0
 */
public abstract class AbstractValueProviderService implements ValueProviderService, Command {

  private static final String DEPLOYMENT = "DEPLOYMENT";

  private LazyValue<ValueProviderCache> valueProviderCache;
  private LazyValue<? extends ComponentLocator<ComponentAst>> locator;
  private Serializer serializer;

  protected AbstractValueProviderService(Serializer serializer,
                                         LazyValue<ValueProviderCache> valueProviderCache,
                                         LazyValue<? extends ComponentLocator<ComponentAst>> locator) {
    requireNonNull(serializer, "serializer cannot be null");
    requireNonNull(valueProviderCache, "value providers cache cannot be null");
    requireNonNull(locator, "component locator should not be null");

    this.serializer = serializer;
    this.valueProviderCache = valueProviderCache;
    this.locator = locator;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueResult getValues(ValueProviderRequest valueProviderRequest) {
    try {
      try {
        return toValueResultDTO(
                                locator
                                    .get()
                                    .get(builderFromStringRepresentation(valueProviderRequest.getLocation()).build())
                                    .map(componentAst -> valueProviderCache.get().getValues(componentAst,
                                                                                            valueProviderRequest
                                                                                                .getProviderName(),
                                                                                            () -> doGetValue(valueProviderRequest)))
                                    .orElse(resultFrom(newFailure().withMessage("Could not find component with location: "
                                        + valueProviderRequest.getLocation()
                                        + " for resolving values").withFailureCode(INVALID_LOCATION).build())));
      } catch (CacheWrappedException e) {
        throw e.getWrappedException();
      }
    } catch (ToolingException e) {
      throw e;
    } catch (Exception e) {
      return toFailureValueResultDTO(e);
    }
  }

  @Override
  public Feature<Action<ValueProviderDisposeRequest>> disposeCachedValues() {
    return enabled(
                   new DispacheableAction<>(
                                            request -> locator
                                                .get()
                                                .get(builderFromStringRepresentation(request.getLocation().toString()).build())
                                                .ifPresent(componentAst -> valueProviderCache.get()
                                                    .dispose(componentAst, request.getParameterNames())),
                                            ValueProviderDisposeRequest.class,
                                            serializer));
  }

  protected abstract org.mule.runtime.api.value.ValueResult doGetValue(ValueProviderRequest valueProviderRequest);

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getValues": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(ValueProviderRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        ValueProviderRequest valueProviderRequest = serializer.deserialize(arguments[0]);
        return serializer.serialize(getValues(valueProviderRequest));
      }
      case "disposeCachedValues": {
        return disposeCachedValues();
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

  private static ValueResult toValueResultDTO(org.mule.runtime.api.value.ValueResult valueResult) {
    HashSet<Value> values = new HashSet<>();
    for (org.mule.runtime.api.value.Value value : valueResult.getValues()) {
      values.add(toValueDTO(value));
    }

    return new ValueResult(values, toResolvingFailureDTO(valueResult.getFailure()));
  }

  private static Value toValueDTO(org.mule.runtime.api.value.Value value) {
    HashSet<Value> children = new HashSet<>();
    for (org.mule.runtime.api.value.Value child : value.getChilds()) {
      children.add(toValueDTO(child));
    }

    return new Value(value.getId(), value.getDisplayName(), value.getPartName(), children);
  }

  private static ResolvingFailure toResolvingFailureDTO(Optional<org.mule.runtime.api.value.ResolvingFailure> resolvingFailureOptional) {
    if (!resolvingFailureOptional.isPresent()) {
      return null;
    }

    org.mule.runtime.api.value.ResolvingFailure resolvingFailure = resolvingFailureOptional.get();

    return new ResolvingFailure(resolvingFailure.getMessage(), resolvingFailure.getReason(), resolvingFailure.getFailureCode());
  }

  private static ValueResult toFailureValueResultDTO(Throwable t) {

    String failureCode = UNKNOWN;
    if (t instanceof CompletionException) {
      if (t.getCause() instanceof DeploymentException) {
        failureCode = DEPLOYMENT;
      }
    }

    return new ValueResult(emptySet(), new ResolvingFailure(t.getMessage(), getStackTrace(t), failureCode));
  }
}
