/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * {@link ExtensionModelService} that use a cache for {@link ExtensionModel}s
 */
public class CachedExtensionModelService extends AbstractMuleRuntimeExtensionModelProvider {

  private ExtensionModelServiceCache extensionModelServiceCache;

  /**
   * Creates a new instance
   *
   * @param mavenClient {@link MavenClient} to resolve artifact descriptors.
   * @param extensionModelServiceCache the cache for {@link ExtensionModel}s
   * @param extensionModelService the service to use to load {@link ExtensionModel}s in case of a cache miss.
   */
  public CachedExtensionModelService(MavenClient mavenClient,
                                     ExtensionModelServiceCache extensionModelServiceCache,
                                     InternalExtensionModelService extensionModelService) {
    super(mavenClient, extensionModelService);
    this.extensionModelServiceCache = extensionModelServiceCache;
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return extensionModelServiceCache
        .loadExtensionInformation(resolveBundleDependency(toBundleDescriptor(pluginDescriptor)), extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<String> getMinMuleVersion(ArtifactDescriptor pluginDescriptor) {
    return extensionModelServiceCache
        .loadExtensionInformation(resolveBundleDependency(toBundleDescriptor(pluginDescriptor)), extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<LoadedExtensionInformation> getExtensionModel(File plugin) {
    return extensionModelService.loadExtensionData(plugin);
  }

  @Override
  public Optional<ExtensionModel> getExtensionModel(BundleDependency bundleDependency) {
    return extensionModelServiceCache.loadExtensionInformation(bundleDependency, extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getExtensionModel());
  }

  @Override
  public Optional<String> getMinMuleVersion(File plugin) {
    return extensionModelService.loadExtensionData(plugin)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getMinMuleVersion());
  }

  @Override
  public Optional<String> getExtensionSchema(File plugin) {
    return extensionModelService.loadExtensionData(plugin)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public Optional<String> getExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return extensionModelServiceCache
        .loadExtensionInformation(resolveBundleDependency(toBundleDescriptor(pluginDescriptor)), extensionModelService)
        .map(loadedExtensionInformation -> loadedExtensionInformation.getSchema().get());
  }

  @Override
  public List<ExtensionModel> getRuntimeExtensionModels() {
    return extensionModelServiceCache.loadRuntimeExtensionModels(extensionModelService);
  }

}
