/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static org.mule.runtime.api.component.location.Location.builderFromStringRepresentation;
import static org.mule.runtime.api.metadata.resolving.FailureCode.COMPONENT_NOT_FOUND;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.api.feature.Feature.enabled;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataKeysContainerDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataResultDTO;

import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.config.api.dsl.model.metadata.ModelBasedMetadataCacheIdGeneratorFactory;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGenerator;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGeneratorFactory;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.feature.Action;
import org.mule.tooling.client.api.feature.Feature;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.internal.action.DispacheableAction;
import org.mule.tooling.client.internal.metadata.MetadataCache;
import org.mule.tooling.client.internal.serialization.Serializer;
import org.mule.tooling.client.internal.utils.ComponentModelLocatorHelper;

import java.util.Optional;

/**
 * Public implementation of {@link MetadataService} that adapts Mule-API objects to DTOs
 *
 * @since 4.0
 */
public class ToolingMetadataServiceAdapter implements MetadataService, Command {

  private LazyValue<ArtifactAst> applicationModel;
  private LazyValue<MetadataCacheIdGeneratorFactory.ComponentLocator<ComponentAst>> componentLocator;
  private LazyValue<MetadataCacheIdGenerator<ComponentAst>> metadataCacheIdGenerator;

  private MetadataProvider metadataProvider;
  private LazyValue<MetadataCache> metadataCache;

  private Serializer serializer;
  private final ComponentModelLocatorHelper componentModelLocatorHelper;

  /**
   * Creates an instance of this service adapter.
   *
   * @param applicationModel    {@link ArtifactAst} to resolve the {@link ComponentAst}. Non null.
   * @param metadataProvider    {@link MetadataProvider} for resolving metadata using Mule Runtime. Non null.
   * @param metadataCache       {@link MetadataCache} to fetch Metadata from cache. Non null.
   * @param serializer          {@link Serializer} to be used for sending a receiving arguments and responses in calls to the {@link org.mule.tooling.client.api.ToolingRuntimeClient} being used
   * @param dslResolvingContext {@link DslResolvingContext} to create a {@link MetadataCacheIdGenerator<ComponentAst>} to use for generating keys in runtime cache
   */
  public ToolingMetadataServiceAdapter(LazyValue<ArtifactAst> applicationModel,
                                       MetadataProvider metadataProvider,
                                       LazyValue<MetadataCache> metadataCache,
                                       Serializer serializer,
                                       LazyValue<DslResolvingContext> dslResolvingContext,
                                       LazyValue<MetadataCacheIdGeneratorFactory.ComponentLocator<ComponentAst>> componentLocator) {
    checkNotNull(applicationModel, "applicationModel cannot be null");
    checkNotNull(metadataProvider, "metadataProvider cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");
    checkNotNull(dslResolvingContext, "dslResolvingContext cannot be null");

    this.applicationModel = applicationModel;
    this.metadataProvider = metadataProvider;
    this.metadataCache = metadataCache;

    this.componentLocator = componentLocator;

    this.serializer = serializer;
    this.componentModelLocatorHelper = new ComponentModelLocatorHelper();

    this.metadataCacheIdGenerator = new LazyValue<>(() -> new ModelBasedMetadataCacheIdGeneratorFactory()
        .create(dslResolvingContext.get(), componentLocator.get()));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public org.mule.tooling.client.api.metadata.MetadataResult<org.mule.tooling.client.api.metadata.MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException {

    Optional<ComponentAst> componentModel =
        componentModelLocatorHelper.findComponentModel(applicationModel.get(), metadataKeysRequest.getLocation());

    MetadataResult<MetadataKeysContainer> metadataKeys;
    if (!componentModel.isPresent()) {
      metadataKeys = failure(newFailure()
          .withMessage(format("No object found at location %s", metadataKeysRequest.getLocation().toString())).withFailureCode(
                                                                                                                               COMPONENT_NOT_FOUND)
          .onKeys());
    } else {
      metadataKeys =
          metadataCache.get().getMetadataKeys(componentModel.get(), () -> metadataProvider.getMetadataKeys(metadataKeysRequest));
    }

    return toMetadataResultDTO(metadataKeys, () -> toMetadataKeysContainerDTO(metadataKeys.get()));
  }

  private void invalidateRuntimeCacheFor(ComponentAst componentAst) {
    metadataCacheIdGenerator.get().getIdForGlobalMetadata(componentAst)
        .ifPresent(metadataCacheId -> metadataProvider.disposeMetadataCache(metadataCacheId.getValue()));
  }

  @Override
  public Feature<Action<org.mule.tooling.client.api.component.location.Location>> disposeMetadataCache() {
    return enabled(
                   new DispacheableAction<>(
                                            location -> {
                                              org.mule.runtime.api.component.location.Location componentLocation =
                                                  builderFromStringRepresentation(location.toString()).build();
                                              ToolingMetadataServiceAdapter.this.componentLocator.get().get(componentLocation)
                                                  .ifPresent(
                                                             componentConfiguration -> {
                                                               invalidateRuntimeCacheFor(componentConfiguration);
                                                               metadataCache.get().dispose(componentConfiguration);
                                                             });
                                            },
                                            org.mule.tooling.client.api.component.location.Location.class,
                                            serializer));
  }

  @Override
  public Feature<Action<Location>> invalidateMetadataCacheKeys() {
    return enabled(
                   new DispacheableAction<>(
                                            location -> {
                                              org.mule.runtime.api.component.location.Location componentLocation =
                                                  builderFromStringRepresentation(location.toString()).build();
                                              ToolingMetadataServiceAdapter.this.componentLocator.get().get(componentLocation)
                                                  .ifPresent(
                                                             componentConfiguration -> {
                                                               invalidateRuntimeCacheFor(componentConfiguration);
                                                               metadataCache.get()
                                                                   .invalidateMetadataKeysFor(componentConfiguration);
                                                             });
                                            },
                                            org.mule.tooling.client.api.component.location.Location.class,
                                            serializer));
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getMetadataKeys": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(MetadataKeysRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serializer.serialize(getMetadataKeys(serializer.deserialize(arguments[0])));
      }
      case "disposeMetadataCache": {
        return disposeMetadataCache();
      }
      case "invalidateMetadataCacheKeys": {
        return invalidateMetadataCacheKeys();
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }
}
