/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.mule.tooling.client.internal.application.RemoteArtifactContextFactory.createRemoteArtifactContext;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.meta.MuleVersion;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.deployment.model.api.DeployableArtifactDescriptor;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import java.io.File;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;

/**
 * Base class for {@link Artifact}.
 *
 * @since 4.1
 */
public abstract class AbstractArtifact<T extends DeployableArtifactDescriptor> implements Artifact {

  private static final Logger LOGGER = getLogger(AbstractArtifact.class);

  private final String id;
  private final Map<String, String> properties;
  protected ToolingArtifactContext context;

  private LazyValue<List<ExtensionModel>> extensionModels;
  private LazyValue<ToolingArtifactClassLoader> artifactClassLoader;
  private LazyValue<ToolingApplicationModel> applicationModel;

  protected ArtifactResources artifactResources;
  protected T artifactDescriptor;
  protected volatile boolean deployed;
  protected volatile String remoteArtifactId;

  protected LazyValue<RuntimeToolingService> runtimeToolingService;
  protected ArtifactDeployer artifactDeployer;

  private final AtomicBoolean disposed = new AtomicBoolean(false);
  protected NoFailureConfigurationProperties noFailureConfigurationProperties =
      new NoFailureConfigurationProperties();

  protected ReadWriteLock deploymentLock = new ReentrantReadWriteLock();

  AbstractArtifact(String id, ArtifactResources artifactResources, T artifactDescriptor, ToolingArtifactContext context,
                   Map<String, String> properties) {
    requireNonNull(id, "id cannot be null");
    requireNonNull(artifactResources, "artifactResources cannot be null");
    requireNonNull(artifactDescriptor, "artifactBundleDescriptor cannot be null");
    requireNonNull(context, "context cannot be null");
    requireNonNull(properties, "properties cannot be null");

    this.id = id;
    this.remoteArtifactId = this.id;
    this.artifactResources = artifactResources;
    this.artifactDescriptor = artifactDescriptor;
    this.context = context;
    this.properties = properties;

    this.applicationModel = newToolingApplicationModelLazyValue();
    this.artifactClassLoader = newToolingArtifactClassLoaderLazyValue();
    this.extensionModels = newExtensionModelsLazyValue();

    this.runtimeToolingService = new LazyValue(() -> context.getRuntimeToolingService());
    this.artifactDeployer = createRemoteArtifactContext(artifactResources, properties, runtimeToolingService);
  }

  @Override
  public String getId() {
    return id;
  }

  @Override
  public ArtifactType getArtifactType() {
    return artifactResources.getArtifactType();
  }

  @Override
  public List<ExtensionModel> getExtensionModels() {
    checkState();
    return extensionModels.get();
  }

  @Override
  public String getArtifactName() {
    checkState();
    return getArtifactClassLoader().getArtifactId();
  }

  @Override
  public File getRootArtifactFolder() {
    checkState();
    return getArtifactClassLoader().getArtifactDescriptor().getRootFolder();
  }

  @Override
  public Map<String, String> getProperties() {
    return properties;
  }

  @Override
  public URL getArtifactUrlContent() {
    checkState();
    return artifactResources.getArtifactUrlContent();
  }

  @Override
  public ToolingArtifactClassLoader getArtifactClassLoader() {
    checkState();
    return artifactClassLoader.get();
  }

  @Override
  public synchronized void setContext(ToolingArtifactContext context) {
    checkState();
    if (this.context != null && this.context.getAgentConfiguration().isPresent() && context.getAgentConfiguration().isPresent()) {
      if (!this.context.getAgentConfiguration().get().getToolingApiUrl()
          .equals(context.getAgentConfiguration().get().getToolingApiUrl())) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Application context has changed, trying to dispose (if present) the remote application");
        }
        disposeRemoteArtifactContext();
      }
    }
    this.context = context;
  }

  @Override
  public void dispose() {
    checkState();
    if (!disposed.getAndSet(true)) {
      disposeClassLoader();
      disposeResources();
      Lock writeLock = deploymentLock.writeLock();
      try {
        writeLock.lock();
        disposeRemoteArtifactContext();
      } finally {
        writeLock.unlock();
      }
    }
  }

  protected void disposeRemoteArtifactContext() {
    if (deployed) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Disposing Mule Runtime remote artifact for artifactId: {}", id);
      }
      try {
        artifactDeployer.dispose(remoteArtifactId);
      } catch (Exception e) {
        // Nothing to do here...
      } finally {
        deployed = false;
        remoteArtifactId = id;
      }
    }
  }

  private void disposeResources() {
    if (this.artifactResources != null) {
      this.artifactResources.dispose();
    }
    this.applicationModel = null;
    this.extensionModels = null;
  }

  private void disposeClassLoader() {
    if (artifactClassLoader != null && artifactClassLoader.isComputed()) {
      try {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Disposing artifact class loader");
        }
        artifactClassLoader.get().dispose();
      } finally {
        artifactClassLoader = null;
      }
    }
  }

  @Override
  public ToolingApplicationModel getApplicationModel() {
    checkState();
    return applicationModel.get();
  }

  @Override
  public MuleVersion getMinMuleVersion() {
    checkState();
    MuleVersion effectiveMinMuleVersion = this.artifactDescriptor.getMinMuleVersion();
    for (ArtifactPluginDescriptor artifactPluginDescriptor : getEffectivePlugins()) {
      MuleVersion candidate = artifactPluginDescriptor.getMinMuleVersion();
      if (candidate.newerThan(effectiveMinMuleVersion)) {
        effectiveMinMuleVersion = candidate;
      }
    }
    return effectiveMinMuleVersion;
  }

  protected List<ArtifactPluginDescriptor> getEffectivePlugins() {
    return ImmutableList.copyOf(this.artifactDescriptor.getPlugins());
  }

  protected abstract LazyValue<ToolingArtifactClassLoader> newToolingArtifactClassLoaderLazyValue();

  protected abstract LazyValue<ToolingApplicationModel> newToolingApplicationModelLazyValue();

  private LazyValue<List<ExtensionModel>> newExtensionModelsLazyValue() {
    return new LazyValue<>(() -> {
      final List<ExtensionModel> extensionModels = getArtifactPluginClassLoaders().stream()
          .map(artifactPluginClassLoader -> createBundleDependency(artifactPluginClassLoader.getArtifactDescriptor()))
          .map(bundleDependency -> context.getMuleRuntimeExtensionModelProvider()
              .getExtensionModel(bundleDependency).orElse(null))
          .filter(extensionModel -> extensionModel != null)
          .collect(toList());
      extensionModels.addAll(context.getMuleRuntimeExtensionModelProvider().getRuntimeExtensionModels());
      return extensionModels;
    });
  }

  private BundleDependency createBundleDependency(ArtifactDescriptor artifactDescriptor) {
    BundleDescriptor bundleDescriptor = artifactDescriptor.getBundleDescriptor();
    try {
      org.mule.maven.client.api.model.BundleDescriptor.Builder bundleDescriptorBuilder =
          new org.mule.maven.client.api.model.BundleDescriptor.Builder();
      bundleDescriptorBuilder
          .setGroupId(bundleDescriptor.getGroupId())
          .setArtifactId(bundleDescriptor.getArtifactId());
      if (bundleDescriptor.getBaseVersion() != null) {
        // If the artifactDescriptor comes from a heavyweight we don't have a baseVersion as no matter if this
        // is an SNAPSHOT or timestamped SNAPSHOT version we cannot rely on the baseVersion resolved when the
        // artifact was packaged as may have a version that is not the one present in the current Maven
        // local repository were Tooling is using to resolve dependencies.
        bundleDescriptorBuilder.setBaseVersion(bundleDescriptor.getBaseVersion());
      }

      bundleDescriptorBuilder
          .setVersion(bundleDescriptor.getVersion())
          .setType(bundleDescriptor.getType())
          .setClassifier(bundleDescriptor.getClassifier().orElse(null));

      return new BundleDependency.Builder()
          .setBundleUri(artifactDescriptor.getClassLoaderModel().getUrls()[0].toURI())
          .sedBundleDescriptor(bundleDescriptorBuilder.build())
          .build();
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  }

  protected List<ArtifactClassLoader> getArtifactPluginClassLoaders() {
    return artifactClassLoader.get().getArtifactPluginClassLoaders();
  }

  @Override
  public String toString() {
    return this.getClass().getSimpleName() + "{artifactName=" + getArtifactName() + ", artifactResources=" + artifactResources
        + "}";
  }

  protected void checkState() {
    Preconditions.checkState(!disposed.get(), "Application already disposed, cannot be used anymore");
  }

  /**
   * Implementation that allows to load an {@link ArtifactAst} without having the properties defined.
   */
  private class NoFailureConfigurationProperties implements ConfigurationProperties {

    @Override
    public <T> Optional<T> resolveProperty(String key) {
      return of((T) "");
    }

    @Override
    public Optional<Boolean> resolveBooleanProperty(String key) {
      return of(false);
    }

    @Override
    public Optional<String> resolveStringProperty(String key) {
      return of("");
    }

  }

}
