/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.lang.String.format;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ComponentModelVisitor;
import org.mule.runtime.api.meta.model.OutputModel;
import org.mule.runtime.api.meta.model.construct.ConstructModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.ast.api.ComponentAst;

import java.util.List;
import java.util.Map;

public class MetadataCachePopulator<T extends ComponentModel> {

  private ComponentAst componentAst;
  private ToolingMetadataCacheIdGenerator cacheIdGenerator;
  private Map<String, MetadataType> metadataTypesCache;
  private T component;

  public MetadataCachePopulator(Map<String, MetadataType> metadataTypesCache,
                                ToolingMetadataCacheIdGenerator cacheIdGenerator,
                                ComponentAst componentAst,
                                T component) {
    this.metadataTypesCache = metadataTypesCache;
    this.cacheIdGenerator = cacheIdGenerator;
    this.componentAst = componentAst;
    this.component = component;
  }

  public void populateCache() {
    component.accept(new ComponentModelVisitor() {

      @Override
      public void visit(ConstructModel constructModel) {
        // Nothing to do
      }

      @Override
      public void visit(OperationModel operationModel) {
        operationModel.getParameterGroupModels().stream().forEach(parameterGroupModel -> {
          populateCacheInputParameters(parameterGroupModel.getParameterModels());
        });

        populateCacheOutput(operationModel.getOutput());

        populateCacheOutputAttributes(operationModel.getOutputAttributes());
      }

      @Override
      public void visit(SourceModel sourceModel) {
        sourceModel.getParameterGroupModels().stream().forEach(parameterGroupModel -> {
          populateCacheInputParameters(parameterGroupModel.getParameterModels());
        });

        //TODO: MULE-17263
        sourceModel.getSuccessCallback();

        //TODO: MULE-17263
        sourceModel.getErrorCallback();

        populateCacheOutput(sourceModel.getOutput());

        populateCacheOutputAttributes(sourceModel.getOutputAttributes());
      }
    });
  }

  private void populateCacheInputParameters(List<ParameterModel> parameterModels) {
    parameterModels.stream().forEach(parameterModel -> {
      if (parameterModel.hasDynamicType()) {
        String idForComponentInputMetadata =
            cacheIdGenerator.getIdForComponentInputMetadata(componentAst, parameterModel.getName())
                .orElseThrow(() -> new IllegalArgumentException(format("Couldn't generate a MetadataCacheId for %s parameter name: %s",
                                                                       componentAst.getLocation(),
                                                                       parameterModel.getName())));
        metadataTypesCache.putIfAbsent(idForComponentInputMetadata, parameterModel.getType());
      }
    });
  }

  private void populateCacheOutput(OutputModel outputModel) {
    if (outputModel.hasDynamicType()) {
      String idForComponentOutputMetadata =
          cacheIdGenerator.getIdForComponentOutputMetadata(componentAst)
              .orElseThrow(() -> new IllegalArgumentException(format("Couldn't generate a MetadataCacheId for %s output",
                                                                     componentAst.getLocation())));
      metadataTypesCache.putIfAbsent(idForComponentOutputMetadata, outputModel.getType());
    }
  }

  private void populateCacheOutputAttributes(OutputModel outputModel) {
    if (outputModel.hasDynamicType()) {
      String idForComponentAttributesMetadata =
          cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)
              .orElseThrow(() -> new IllegalArgumentException(format("Couldn't generate a MetadataCacheId for %s output attributes",
                                                                     componentAst.getLocation())));
      metadataTypesCache.putIfAbsent(idForComponentAttributesMetadata, outputModel.getType());
    }
  }
}
