/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.util.Optional.of;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.sameInstance;
import static org.junit.Assert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.metadata.api.model.MetadataType;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheId;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGenerator;
import org.mule.runtime.extension.api.persistence.metadata.ComponentResultJsonSerializer;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import com.google.common.collect.ImmutableList;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.junit.MockitoJUnitRunner;
import org.skyscreamer.jsonassert.JSONAssert;

@RunWith(MockitoJUnitRunner.class)
public class ComponentModelMediatorTestCase extends AbstractMetadataTestCase {

  private static final String KEY = "testKey";

  // OperationModel
  @Test
  public void outputMetadataNotPresentInCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(KEY));

    Optional<OperationModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(false));
  }

  @Test
  public void outputMetadataPresentInCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    Optional<OperationModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    OperationModel resolvedOperationModel = enrichedComponentModelOptional.get();

    assertThat(resolvedOperationModel.getOutput().getType(), sameInstance(objectType));
    assertThat(resolvedOperationModel.getOutputAttributes().getType(), sameInstance(stringType));
    assertThat(resolvedOperationModel.getParameterGroupModels().get(0).getParameter(STATIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(stringType));
  }

  @Test
  public void outputAttributesMetadataNotPresentInCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    Optional<OperationModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(false));
  }

  @Test
  public void outputAttributesMetadataPresentInCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    Optional<OperationModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    OperationModel resolvedOperationModel = enrichedComponentModelOptional.get();

    assertThat(resolvedOperationModel.getOutputAttributes().getType(), sameInstance(objectType));
    assertThat(resolvedOperationModel.getOutput().getType(), sameInstance(stringType));
    assertThat(resolvedOperationModel.getParameterGroupModels().get(0).getParameter(STATIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(stringType));
  }

  @Test
  public void parameterMetadataPresentInCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel, dynamicParameterModel), staticOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentInputMetadata(componentAst, dynamicParameterModel.getName()))
        .thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    Optional<OperationModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    OperationModel resolvedOperationModel = enrichedComponentModelOptional.get();

    assertThat(resolvedOperationModel.getParameterGroupModels().get(0).getParameter(DYNAMIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(objectType));
    assertThat(resolvedOperationModel.getParameterGroupModels().get(0).getParameter(STATIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(stringType));
    assertThat(resolvedOperationModel.getOutput().getType(), sameInstance(stringType));
    assertThat(resolvedOperationModel.getOutputAttributes().getType(), sameInstance(stringType));
  }

  @Test
  public void parameterMetadataNotPresentInCacheForOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel, dynamicParameterModel), staticOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentInputMetadata(componentAst, dynamicParameterModel.getName()))
        .thenReturn(of(KEY));

    Optional<OperationModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(false));
  }

  @Test
  public void copyOperationModel() {
    OperationModel operationModel =
        getOperationModel(ImmutableList.of(staticParameterModel), staticOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    Optional<OperationModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, operationModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    JSONAssert.assertEquals(toJson(operationModel), toJson(enrichedComponentModelOptional.get()), true);
  }

  private <T extends ComponentModel> String toJson(T componentModel) {
    ComponentResultJsonSerializer<T> componentModelComponentResultJsonSerializer = new ComponentResultJsonSerializer<>(true);

    ComponentMetadataDescriptor<T> componentMetadataDescriptor = ComponentMetadataDescriptor.builder(componentModel).build();
    MetadataResult<ComponentMetadataDescriptor<T>> componentMetadataDescriptorMetadataResult =
        MetadataResult.success(componentMetadataDescriptor);
    return componentModelComponentResultJsonSerializer.serialize(componentMetadataDescriptorMetadataResult);
  }

  // SourceModel
  @Test
  public void copySourceModel() {
    SourceModel sourceModel =
        getSourceModel(ImmutableList.of(staticParameterModel), staticOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    Optional<SourceModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    JSONAssert.assertEquals(toJson(sourceModel), toJson(enrichedComponentModelOptional.get()), true);
  }

  @Test
  public void outputMetadataNotPresentInCacheForSourceModel() {
    SourceModel sourceModel = getSourceModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(KEY));

    Optional<SourceModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(false));
  }

  @Test
  public void outputMetadataPresentInCacheForSourceModel() {
    SourceModel sourceModel = getSourceModel(ImmutableList.of(staticParameterModel), dynamicOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentOutputMetadata(componentAst)).thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    Optional<SourceModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    SourceModel resolvedSourceModel = enrichedComponentModelOptional.get();

    assertThat(resolvedSourceModel.getOutput().getType(), sameInstance(objectType));
    assertThat(resolvedSourceModel.getOutputAttributes().getType(), sameInstance(stringType));
    assertThat(resolvedSourceModel.getParameterGroupModels().get(0).getParameter(STATIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(stringType));
  }

  @Test
  public void outputAttributesMetadataNotPresentInCacheForSourceModel() {
    SourceModel sourceModel =
        getSourceModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    Optional<SourceModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(false));
  }

  @Test
  public void outputAttributesMetadataPresentInCacheForSourceModel() {
    SourceModel sourceModel =
        getSourceModel(ImmutableList.of(staticParameterModel), staticOutputModel, dynamicOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentAttributesMetadata(componentAst)).thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    Optional<SourceModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    SourceModel resolvedSourceModel = enrichedComponentModelOptional.get();

    assertThat(resolvedSourceModel.getOutputAttributes().getType(), sameInstance(objectType));
    assertThat(resolvedSourceModel.getOutput().getType(), sameInstance(stringType));
    assertThat(resolvedSourceModel.getParameterGroupModels().get(0).getParameter(STATIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(stringType));
  }

  @Test
  public void parameterMetadataPresentInCacheForSourceModel() {
    SourceModel sourceModel =
        getSourceModel(ImmutableList.of(staticParameterModel, dynamicParameterModel), staticOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentInputMetadata(componentAst, dynamicParameterModel.getName()))
        .thenReturn(of(KEY));

    metadataTypesCache.put(KEY, objectType);

    Optional<SourceModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(true));
    SourceModel resolvedSourceModel = enrichedComponentModelOptional.get();

    assertThat(resolvedSourceModel.getParameterGroupModels().get(0).getParameter(DYNAMIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(objectType));
    assertThat(resolvedSourceModel.getParameterGroupModels().get(0).getParameter(STATIC_PARAMETER_MODEL_NAME).get().getType(),
               sameInstance(stringType));
    assertThat(resolvedSourceModel.getOutput().getType(), sameInstance(stringType));
    assertThat(resolvedSourceModel.getOutputAttributes().getType(), sameInstance(stringType));
  }

  @Test
  public void parameterMetadataNotPresentInCacheForSourceModel() {
    SourceModel sourceModel =
        getSourceModel(ImmutableList.of(staticParameterModel, dynamicParameterModel), staticOutputModel, staticOutputModel);

    Map<String, MetadataType> metadataTypesCache = new HashMap<>();
    ToolingMetadataCacheIdGenerator cacheIdGenerator = mock(ToolingMetadataCacheIdGenerator.class);
    ComponentAst componentAst = mock(ComponentAst.class);

    when(cacheIdGenerator.getIdForComponentInputMetadata(componentAst, dynamicParameterModel.getName()))
        .thenReturn(of(KEY));

    Optional<SourceModel> enrichedComponentModelOptional =
        new ComponentModelMediator<>(metadataTypesCache, cacheIdGenerator, componentAst, sourceModel)
            .enrichComponentModel();

    assertThat(enrichedComponentModelOptional.isPresent(), is(false));
  }

}
