/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.metadata;

import static java.util.Collections.emptyMap;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.mockito.Mockito.when;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newListValue;
import static org.mule.runtime.app.declaration.api.fluent.ElementDeclarer.newParameterGroup;
import static org.mule.runtime.core.api.extension.MuleExtensionModelProvider.MULE_NAME;
import static org.mule.runtime.internal.dsl.DslConstants.FLOW_ELEMENT_IDENTIFIER;

import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.app.declaration.api.ArtifactDeclaration;
import org.mule.runtime.app.declaration.api.fluent.ElementDeclarer;
import org.mule.runtime.ast.api.ComponentAst;
import org.mule.runtime.config.api.dsl.processor.ArtifactConfig;
import org.mule.runtime.config.dsl.model.AbstractDslModelTestCase;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.config.internal.model.ComponentModel;
import org.mule.runtime.core.internal.metadata.cache.MetadataCacheIdGeneratorFactory;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.junit.Before;
import org.junit.Test;

public class ToolingMetadataCacheIdGeneratorTestCase extends AbstractDslModelTestCase {

  private static final String MY_CONFIG = "myConfig";
  private static final String MY_FLOW = "myFlow";
  private static final Location OPERATION_LOCATION =
      Location.builder().globalName(MY_FLOW).addProcessorsPart().addIndexPart(0).build();

  private ElementDeclarer declarer;
  private Set<ExtensionModel> extensions;
  private DslResolvingContext dslResolvingContext;

  @Before
  public void setUp() {
    extensions = new HashSet<>();
    extensions.add(mockExtension);
    dslResolvingContext = DslResolvingContext.getDefault(extensions);
    declarer = ElementDeclarer.forExtension(EXTENSION_NAME);
  }

  @Test
  public void differentMajorGivesDifferentIds() throws Exception {
    ArtifactDeclaration declaration = getBaseApp();
    final String version1 = "1.0.0";
    when(mockExtension.getVersion()).thenReturn(version1);
    String cacheId1 = getIdForOperation(declaration);
    final String version2 = "2.0.0";
    when(mockExtension.getVersion()).thenReturn(version2);
    String cacheId2 = getIdForOperation(declaration);
    assertThat(cacheId1, is(not(cacheId2)));
  }

  @Test
  public void sameMajorGivesSameId() throws Exception {
    ArtifactDeclaration declaration = getBaseApp();
    final String version1 = "1.0.0";
    when(mockExtension.getVersion()).thenReturn(version1);
    String cacheId1 = getIdForOperation(declaration);
    final String version2 = "1.5.0";
    when(mockExtension.getVersion()).thenReturn(version2);
    String cacheId2 = getIdForOperation(declaration);
    assertThat(cacheId1, is(cacheId2));
  }

  @Test
  public void malformedVersionGivesDelegateId() throws Exception {
    ArtifactDeclaration declaration = getBaseApp();
    final String version = "this is not a valid semver format";
    when(mockExtension.getVersion()).thenReturn(version);
    getIdForOperation(declaration); //does not fail
  }

  private String getIdForOperation(ArtifactDeclaration declaration) throws Exception {
    ApplicationModel app = loadApplicationModel(declaration);
    ComponentAst component = new Locator(app)
        .get(OPERATION_LOCATION)
        .get();
    return createGenerator(app).getIdForComponentMetadata(component).get();
  }

  private ToolingMetadataCacheIdGenerator createGenerator(ApplicationModel app) {
    return new ToolingMetadataCacheIdGenerator(dslResolvingContext, new Locator(app));
  }

  private ArtifactDeclaration getBaseApp() {
    return ElementDeclarer.newArtifact()
        .withGlobalElement(declarer.newConfiguration(CONFIGURATION_NAME)
            .withRefName(MY_CONFIG)
            .withParameterGroup(newParameterGroup()
                .withParameter(CONTENT_NAME, CONTENT_VALUE)
                .withParameter(BEHAVIOUR_NAME, BEHAVIOUR_VALUE)
                .withParameter(LIST_NAME, newListValue().withValue(ITEM_VALUE).build())
                .getDeclaration())
            .withConnection(declarer.newConnection(CONNECTION_PROVIDER_NAME)
                .withParameterGroup(newParameterGroup()
                    .withParameter(CONTENT_NAME, CONTENT_VALUE)
                    .withParameter(BEHAVIOUR_NAME, BEHAVIOUR_VALUE)
                    .withParameter(LIST_NAME,
                                   newListValue().withValue(ITEM_VALUE).build())
                    .getDeclaration())
                .getDeclaration())
            .getDeclaration())
        .withGlobalElement(ElementDeclarer.forExtension(MULE_NAME)
            .newConstruct(FLOW_ELEMENT_IDENTIFIER)
            .withRefName(MY_FLOW)
            .withComponent(
                           declarer.newOperation(OPERATION_NAME)
                               .withConfig(MY_CONFIG)
                               .withParameterGroup(g -> g.withParameter(CONTENT_NAME, "nonKey"))
                               .getDeclaration())
            .getDeclaration())
        .getDeclaration();
  }

  private ApplicationModel loadApplicationModel(ArtifactDeclaration declaration) throws Exception {
    return new ApplicationModel(new ArtifactConfig.Builder().build(),
                                declaration, extensions, emptyMap(), Optional.empty(), Optional.empty(),
                                uri -> getClass().getResourceAsStream(uri));
  }

  private static class Locator implements MetadataCacheIdGeneratorFactory.ComponentLocator<ComponentAst> {

    private final Map<Location, ComponentModel> components = new HashMap<>();

    Locator(ApplicationModel app) {
      app.getRootComponentModel().getInnerComponents().forEach(this::addComponent);
    }

    @Override
    public Optional<ComponentAst> get(Location location) {
      return Optional.ofNullable(components.get(location)).map(cm -> (ComponentAst) cm);
    }

    private Location getLocation(ComponentModel component) {
      return Location.builderFromStringRepresentation(component.getComponentLocation().getLocation()).build();
    }

    private void addComponent(ComponentModel component) {
      components.put(getLocation(component), component);
      component.getInnerComponents().forEach(this::addComponent);
    }

  }
}
