/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.ast.api.ArtifactAst;
import org.mule.runtime.ast.api.xml.AstXmlParser;
import org.mule.runtime.deployment.model.api.DeployableArtifactDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.UnaryOperator;

/**
 * Default implementation for {@link ApplicationModelFactory}.
 *
 * @since 4.0
 */
public class DefaultApplicationModelFactory implements ApplicationModelFactory {

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ArtifactAst> createApplicationModel(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                      Set<ExtensionModel> extensionModels,
                                                      ClassLoader applicationClassLoader,
                                                      Function<ArtifactAst, UnaryOperator<String>> configurationPropertiesResolver,
                                                      Map<String, String> deploymentProperties) {
    final AstXmlParser astXmlParser = AstXmlParser.builder()
        .withSchemaValidationsDisabled()
        .withExtensionModels(extensionModels)
        .withPropertyResolver(propertyKey -> deploymentProperties.getOrDefault(propertyKey,
                                                                               propertyKey))
        .build();

    return createApplicationModel(deployableArtifactDescriptor, astXmlParser, applicationClassLoader,
                                  configurationPropertiesResolver);
  }

  @Override
  public Optional<ArtifactAst> createApplicationModel(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                      AstXmlParser parser, ClassLoader applicationClassLoader,
                                                      Function<ArtifactAst, UnaryOperator<String>> configurationPropertiesResolver) {
    try {
      return of(loadApplicationModel(deployableArtifactDescriptor, parser, applicationClassLoader,
                                     configurationPropertiesResolver));
    } catch (Exception e) {
      throw new ToolingException("Error while creating application model", e);
    }
  }

  /**
   * Loads {@link ArtifactAst} from an {@link InputStream} containing the XML app config.
   *
   * @param deployableArtifactDescriptor  descriptor for the artifact.
   * @param astXmlParser                  the XML parser to be used for generating the artifact model.
   * @param applicationClassLoader        {@link ClassLoader} to resolve resources.
   * @param parentConfigurationProperties {@link ConfigurationProperties} for artifact if present.
   * @param deploymentProperties          {@link Map} with the deployment properties for the artifact.
   * @return a {@link ArtifactAst}.
   */
  private ArtifactAst loadApplicationModel(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                           AstXmlParser astXmlParser,
                                           ClassLoader applicationClassLoader,
                                           Function<ArtifactAst, UnaryOperator<String>> configurationPropertiesResolver) {
    final List<Pair<String, InputStream>> appXmlConfigInputStreams = deployableArtifactDescriptor.getConfigResources().stream()
        .map(configFile -> {
          URL url = applicationClassLoader.getResource(configFile);
          if (url == null) {
            throw new ToolingException(format("Configuration file '%s' cannot be found", configFile));
          }
          try {
            return new Pair<>(configFile, url.openStream());
          } catch (IOException e) {
            throw new ToolingException(format("Error while reading configuration file '%s'", configFile), e);
          }
        })
        .collect(toList());

    return withContextClassLoader(applicationClassLoader,
                                  () -> {
                                    final ArtifactAst ast = astXmlParser.parse(appXmlConfigInputStreams);

                                    final UnaryOperator<String> resolver = configurationPropertiesResolver.apply(ast);

                                    try {
                                      initialiseIfNeeded(resolver);
                                    } catch (InitialisationException e) {
                                      throw new MuleRuntimeException(e);
                                    }
                                    ast.updatePropertiesResolver(resolver);

                                    return ast;
                                  });
  }

}
