/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;
import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.lang.System.nanoTime;
import static java.nio.charset.Charset.defaultCharset;
import static java.util.Collections.emptyList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.concurrent.TimeUnit.NANOSECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.maven.artifact.ArtifactUtils.isSnapshot;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.tooling.client.internal.utils.ServiceUtils.executeHandlingException;

import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.util.PropertiesUtils;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheStats;
import org.apache.commons.io.filefilter.AndFileFilter;
import org.apache.commons.io.filefilter.DirectoryFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.NameFileFilter;
import org.apache.commons.io.filefilter.NotFileFilter;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * In memory cache for {@link ExtensionModel}s. Supports mule-plugin release and snapshot artifacts. It can be enabled to persist
 * the value into disk. This can be done by setting a system property
 * {@link #TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISKSTORE_PATH} to define the path were data will be stored. <br/>
 * The support for SNAPSHOTS is based on the fimestamped version of artifacts (when downloaded from remote repositories) or the
 * {@link File#lastModified()} value for SNAPSHOTS locally installed. Whenever a new snapshot version is discovered any previous
 * version, if it was written into disk, will be removed and invalidated from the memory cache.
 */
public class ExtensionModelServiceCache implements Disposable {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE =
      "tooling.client.ExtensionModelServiceCache.maximumSize";
  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS =
      "tooling.client.ExtensionModelServiceCache.expireAfterAccess";
  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISKSTORE_PATH =
      "tooling.client.ExtensionModelServiceCache.diskStore.path";

  private static final Logger LOGGER = LoggerFactory.getLogger(ExtensionModelServiceCache.class);
  private static final String DEFAULT_MAX_SIZE = "100";

  private static final String MIN_MULE_VERSION = "minMuleVersion";
  private static final String EXTENSION_MODEL = "extensionModel";
  public static final String METADATA_KEY = "metadataKey";
  public static final String METADATA_CACHE_PROPERTIES_FILE_NAME = "metadata-cache.properties";

  private File diskStorePath;

  private List<ExtensionModel> runtimeExtensionModels;

  private Cache<String, Optional<LoadedExtensionInformation>> extensionModelsByArtifact;

  private Integer cacheMaximumSize() {
    Integer cacheSize = valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE, DEFAULT_MAX_SIZE));
    checkArgument(cacheSize > 0,
                  format("Wrong value %d provided in system property %s, the cache cannot be less that zero",
                         cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE));
    return cacheSize;
  }

  private Optional<Long> cacheExpireAfterAccess() {
    if (getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS) == null) {
      return empty();
    }

    Long cacheSize = Long.valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS));
    checkArgument(cacheSize > 0,
                  format("Wrong value %d provided in system property %s, cacheExpireAfterAccess cannot be less that zero",
                         cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE));
    return of(cacheSize);
  }

  private File diskStorePath() {
    if (getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISKSTORE_PATH) == null) {
      return null;
    }

    File diskStorePath = new File(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_DISKSTORE_PATH));
    if (!diskStorePath.exists()) {
      boolean created = diskStorePath.mkdirs();
      checkArgument(created,
                    format("Couldn't create folders Extension Model service cache persistent data using path",
                           diskStorePath));
    }
    return diskStorePath;
  }

  public ExtensionModelServiceCache(final String toolingVersion) {
    this(toolingVersion, false);
  }

  public ExtensionModelServiceCache(final String toolingVersion, boolean recordStats) {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Initialising Extension Model Service cache");
    }

    CacheBuilder builder = CacheBuilder.newBuilder().maximumSize(cacheMaximumSize());
    cacheExpireAfterAccess().ifPresent(value -> builder.expireAfterAccess(value, SECONDS));
    if (recordStats) {
      builder.recordStats();
    }

    this.diskStorePath = diskStorePath();
    if (this.diskStorePath != null) {
      this.diskStorePath = new File(this.diskStorePath, toolingVersion);
      if (!this.diskStorePath.exists()) {
        boolean created = this.diskStorePath.mkdirs();
        checkArgument(created,
                      format("Couldn't create folders Extension Model service cache persistent data using path",
                             this.diskStorePath));
      }
      if (LOGGER.isInfoEnabled()) {
        LOGGER.info("Enabling cache persistence for Extension Model Service on path '{}'", this.diskStorePath.getAbsolutePath());
      }
    } else {
      if (LOGGER.isInfoEnabled()) {
        LOGGER.info("Extension Model Service cache working in memory only");
      }
    }
    extensionModelsByArtifact = builder.build();
  }

  /**
   * Loads an extension model from a {@link ArtifactDescriptor}.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param bundleDependency      the plugin {@link BundleDependency}
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the {@link LoadedExtensionInformation}, empty if the plugin does not have an extension model
   */
  public Optional<LoadedExtensionInformation> loadExtensionInformation(BundleDependency bundleDependency,
                                                                       InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> extensionModelsByArtifact
        .get(newCacheKey(bundleDependency), generateExtensionModelInformation(bundleDependency,
                                                                              () -> extensionModelService
                                                                                  .loadExtensionData(bundleDependency
                                                                                      .getDescriptor()))));
  }

  /**
   * Loads the set of extension models from the list of {@link ArtifactClassLoader}s.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param artifactPluginClassLoaders the list of plugin {@link ArtifactClassLoader}
   * @param extensionModelService      the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return a {@link Set} of {@link LoadedExtensionInformation}, empty if none of the plugins have an extension model
   */
  public Set<ExtensionModel> loadExtensionModels(List<ArtifactClassLoader> artifactPluginClassLoaders,
                                                 InternalExtensionModelService extensionModelService) {
    Set<ExtensionModel> extensionModels = newHashSet();
    for (ArtifactClassLoader artifactClassLoader : artifactPluginClassLoaders) {
      BundleDependency bundleDependency = createBundleDependency(artifactClassLoader.getArtifactDescriptor());
      Optional<LoadedExtensionInformation> loadedExtensionInformation = executeHandlingException(() -> extensionModelsByArtifact
          .get(newCacheKey(bundleDependency), generateExtensionModelInformation(bundleDependency, () -> extensionModelService
              .loadExtensionData(artifactClassLoader, artifactPluginClassLoaders))));
      loadedExtensionInformation.ifPresent(modelInformation -> extensionModels.add(modelInformation.getExtensionModel()));
    }
    return extensionModels;
  }

  private Callable<Optional<LoadedExtensionInformation>> generateExtensionModelInformation(BundleDependency bundleDependency,
                                                                                           Supplier<Optional<LoadedExtensionInformation>> modelLoader) {
    return () -> {
      File pluginFile = getBundleDependencyFile(bundleDependency);
      BundleDescriptor bundleDescriptor = bundleDependency.getDescriptor();
      return readExtensionModelData(pluginFile, bundleDescriptor).orElseGet(() -> {
        Optional<LoadedExtensionInformation> extensionInformationOptional = modelLoader.get();
        extensionInformationOptional.ifPresent(extensionInformation -> {
          if (diskStorePath != null) {
            String cacheKey = newCacheKey(bundleDependency);
            try {
              File extensionModelPersistenceDirectory = getExtensionModelPersistenceDirectory(cacheKey);
              collectPreviousKeys(bundleDescriptor, pluginFile).stream()
                  .forEach(keyToBeInvalidated -> {
                    extensionModelsByArtifact.invalidate(keyToBeInvalidated);
                    deleteQuietly(getExtensionModelPersistenceDirectory(keyToBeInvalidated));
                  });
              Properties content = new Properties();
              content.setProperty(METADATA_KEY, cacheKey);

              writeCacheMetadata(bundleDescriptor, extensionModelPersistenceDirectory, content);
              writeExtensionModelToDisk(bundleDescriptor, pluginFile, extensionInformation);
            } catch (Exception e) {
              if (LOGGER.isWarnEnabled()) {
                LOGGER.warn(
                            "Error while writing {} Extension Model to disk, error: '{}'. Using in-memory cache without persistence for this entry",
                            bundleDescriptor, e.getMessage());
              }
              deleteQuietly(getExtensionModelPersistenceDirectory(cacheKey));
            }
          }
        });
        return extensionInformationOptional;
      });
    };
  }

  private Optional<Optional<LoadedExtensionInformation>> readExtensionModelData(File pluginFile,
                                                                                BundleDescriptor bundleDescriptor) {
    if (diskStorePath != null && existExtensionDataFiles(bundleDescriptor, pluginFile)) {
      try {
        return of(loadExtensionModelFromDisk(bundleDescriptor, pluginFile));
      } catch (Exception e) {
        if (LOGGER.isWarnEnabled()) {
          LOGGER.warn(
                      "Error while loading {} Extension Model from file data, error: '{}'. It will be loaded again from jar file and writing to disk again",
                      bundleDescriptor, e.getMessage());
        }
      }
    }
    return empty();
  }

  private boolean existExtensionDataFiles(BundleDescriptor bundleDescriptor, File pluginFile) {
    return getExtensionModelPersistenceFile(bundleDescriptor, pluginFile).exists() &&
        getExtensionSchemaPersistenceFile(bundleDescriptor, pluginFile).exists();
  }

  private String newCacheKey(BundleDependency bundleDependency) {
    return getPathForBundleDescriptor(bundleDependency.getDescriptor(), getBundleDependencyFile(bundleDependency), empty());
  }

  private File getBundleDependencyFile(BundleDependency bundleDependency) {
    try {
      return toFile(bundleDependency.getBundleUri().toURL());
    } catch (MalformedURLException e) {
      throw new RuntimeException(format("Error while getting artifact file for %s", bundleDependency), e);
    }
  }

  private Optional<LoadedExtensionInformation> loadExtensionModelFromDisk(BundleDescriptor bundleDescriptor, File pluginFile) {
    long startTime = nanoTime();
    try {
      File extensionModelFile = getExtensionModelPersistenceFile(bundleDescriptor, pluginFile);
      String content = readFileToString(extensionModelFile, defaultCharset());

      JSONObject jsonObject = new JSONObject(content);
      if (jsonObject.length() != 0) {
        if (jsonObject.has(MIN_MULE_VERSION) && jsonObject.has(EXTENSION_MODEL)) {
          ExtensionModel extensionModel =
              new ExtensionModelJsonSerializer().deserialize(jsonObject.get(EXTENSION_MODEL).toString());
          return of(new LoadedExtensionInformation(extensionModel, new LazyValue<>(() -> {
            File schemaFile = getExtensionSchemaPersistenceFile(bundleDescriptor, pluginFile);
            if (!schemaFile.exists()) {
              if (LOGGER.isInfoEnabled()) {
                LOGGER.info("No schema file found from disk cache for {}", bundleDescriptor);
              }
              return "";
            }
            try {
              return readFileToString(schemaFile, defaultCharset());
            } catch (IOException e) {
              throw new UncheckedIOException(e);
            }
          }), jsonObject.getString(MIN_MULE_VERSION)));
        } else {
          if (LOGGER.isWarnEnabled()) {
            LOGGER.warn("Ignoring file for {} as its content is invalid", bundleDescriptor);
          }
        }
      }
      return empty();
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    } finally {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Extension model for {} loaded from file disk in {}ms", bundleDescriptor,
                     NANOSECONDS.toMillis(nanoTime() - startTime));
      }
    }
  }

  private List<String> collectPreviousKeys(BundleDescriptor bundleDescriptor, File pluginFile) {
    if (!isSnapshot(bundleDescriptor.getVersion())) {
      return emptyList();
    }
    List<String> keysToBeInvalidated = newArrayList();
    IOFileFilter filter = DirectoryFileFilter.INSTANCE;
    if (!bundleDescriptor.getVersion().equals(bundleDescriptor.getBaseVersion())) {
      filter = new AndFileFilter(filter, new NotFileFilter(new NameFileFilter(bundleDescriptor.getVersion())));
    }
    File[] directories = getExtensionModelPersistenceFile(bundleDescriptor, pluginFile).getParentFile().getParentFile()
        .listFiles((FilenameFilter) filter);
    if (directories != null) {
      for (File directory : directories) {
        keysToBeInvalidated.add(readCacheMetadata(directory).getProperty(METADATA_KEY));
      }
    }
    return keysToBeInvalidated;
  }

  private Properties readCacheMetadata(File directory) {
    try (InputStream inputStream = new FileInputStream(new File(directory, METADATA_CACHE_PROPERTIES_FILE_NAME))) {
      return PropertiesUtils.loadProperties(inputStream);
    } catch (IOException e) {
      throw new UncheckedIOException(format("Error while reading cache entry metadata from %s", directory), e);
    }
  }

  private void writeCacheMetadata(BundleDescriptor bundleDescriptor, File directory, Properties content) {
    if (!isSnapshot(bundleDescriptor.getVersion())) {
      return;
    }

    if (!directory.exists()) {
      directory.mkdirs();
    }

    File cacheMetadataFile = new File(directory, METADATA_CACHE_PROPERTIES_FILE_NAME);
    try (OutputStream outputStream = new FileOutputStream(cacheMetadataFile, false)) {
      content.store(outputStream, "Writing metadata file for cache entry");
    } catch (IOException e) {
      if (LOGGER.isWarnEnabled()) {
        LOGGER.warn(format("Couldn't write to disk Extension Model cache metadata '{}'", cacheMetadataFile.getAbsolutePath()), e);
      }
      if (cacheMetadataFile.exists()) {
        deleteQuietly(cacheMetadataFile);
      }
    }
  }

  private File getExtensionModelPersistenceDirectory(String key) {
    return diskStorePath.toPath().resolve(key).toFile();
  }

  private String getExtensionModelPersistenceDirectoryPath(BundleDescriptor bundleDescriptor, File pluginFile) {
    StringBuilder path = new StringBuilder(128);
    path.append(bundleDescriptor.getGroupId().replace('.', '/')).append('/');
    path.append(bundleDescriptor.getArtifactId()).append('/');
    path.append(bundleDescriptor.getVersion());
    if (isNormalizedSnapshotVersion(bundleDescriptor)) {
      path.append("-").append(pluginFile.lastModified());
    }
    return path.toString();
  }

  private File getExtensionModelPersistenceFile(BundleDescriptor bundleDescriptor, File pluginFile) {
    return diskStorePath.toPath().resolve(getPathForBundleDescriptor(bundleDescriptor, pluginFile, of("json"))).toFile();
  }

  private File getExtensionSchemaPersistenceFile(BundleDescriptor bundleDescriptor, File pluginFile) {
    return diskStorePath.toPath().resolve(getPathForBundleDescriptor(bundleDescriptor, pluginFile, of("xsd"))).toFile();
  }

  private void writeExtensionModelToDisk(BundleDescriptor bundleDescriptor, File pluginFile,
                                         LoadedExtensionInformation extensionInformation) {
    File extensionModelFile = getExtensionModelPersistenceFile(bundleDescriptor, pluginFile);
    File extensionSchemaFile = getExtensionSchemaPersistenceFile(bundleDescriptor, pluginFile);
    long startTime = nanoTime();
    JSONObject jsonObject = new JSONObject();
    jsonObject.put(MIN_MULE_VERSION, extensionInformation.getMinMuleVersion());
    jsonObject.put(EXTENSION_MODEL,
                   new JSONObject(new ExtensionModelJsonSerializer(false).serialize(extensionInformation.getExtensionModel())));
    try {
      writeStringToFile(extensionModelFile, jsonObject.toString(2), defaultCharset(), false);
      writeStringToFile(extensionSchemaFile, extensionInformation.getSchema().get(), defaultCharset(), false);
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Extension model data for {} written to file disk in {}ms", bundleDescriptor,
                     NANOSECONDS.toMillis(nanoTime() - startTime));
      }
    } catch (IOException e) {
      if (LOGGER.isWarnEnabled()) {
        LOGGER.warn(format("Couldn't write to disk data for Extension Model '{}'", bundleDescriptor), e);
      }
      if (extensionModelFile.exists()) {
        deleteQuietly(extensionModelFile);
      }
      if (extensionSchemaFile.exists()) {
        deleteQuietly(extensionSchemaFile);
      }
    }
  }

  private String getPathForBundleDescriptor(BundleDescriptor bundleDescriptor, File pluginFile,
                                            Optional<String> extensionOptional) {
    StringBuilder path = new StringBuilder(getExtensionModelPersistenceDirectoryPath(bundleDescriptor, pluginFile));
    extensionOptional.ifPresent(extension -> {
      path.append('/');
      path.append(bundleDescriptor.getArtifactId()).append('-');
      path.append(bundleDescriptor.getVersion());
      if (bundleDescriptor.getClassifier().isPresent()) {
        path.append('-').append(bundleDescriptor.getClassifier().get());
      }
      path.append('.').append(extension);
    });
    return path.toString();
  }

  private boolean isNormalizedSnapshotVersion(BundleDescriptor bundleDescriptor) {
    return isSnapshot(bundleDescriptor.getVersion()) && bundleDescriptor.getVersion().equals(bundleDescriptor.getBaseVersion());
  }


  /**
   * Loads an extension model from a {@link File} pointing to the plugin artifact.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param plugin                the plugin descriptor
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the {@link LoadedExtensionInformation}, empty if the plugin does not have an extension model
   */
  public Optional<LoadedExtensionInformation> loadExtensionInformation(File plugin,
                                                                       InternalExtensionModelService extensionModelService) {
    BundleDescriptor bundleDescriptor = extensionModelService.readBundleDescriptor(plugin);
    return loadExtensionInformation(new BundleDependency.Builder()
        .sedBundleDescriptor(bundleDescriptor)
        .setBundleUri(plugin.toURI())
        .build(),
                                    extensionModelService);
  }

  /**
   * Loads the core extension model.
   *
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the core extension model
   */
  public List<ExtensionModel> loadRuntimeExtensionModels(InternalExtensionModelService extensionModelService) {
    if (runtimeExtensionModels == null) {
      runtimeExtensionModels = extensionModelService.loadRuntimeExtensionModels();
    }
    return runtimeExtensionModels;
  }

  @Override
  public void dispose() {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Disposing Extension Model Service cache");
    }
    this.extensionModelsByArtifact.invalidateAll();
  }

  public CacheStats getCacheStats() {
    return extensionModelsByArtifact.stats();
  }

  private BundleDependency createBundleDependency(org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor artifactDescriptor) {
    org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor bundleDescriptor = artifactDescriptor.getBundleDescriptor();
    try {
      org.mule.maven.client.api.model.BundleDescriptor.Builder bundleDescriptorBuilder =
          new org.mule.maven.client.api.model.BundleDescriptor.Builder();
      bundleDescriptorBuilder
          .setGroupId(bundleDescriptor.getGroupId())
          .setArtifactId(bundleDescriptor.getArtifactId());
      if (bundleDescriptor.getBaseVersion() != null) {
        // If the artifactDescriptor comes from a heavyweight we don't have a baseVersion as no matter if this
        // is an SNAPSHOT or timestamped SNAPSHOT version we cannot rely on the baseVersion resolved when the
        // artifact was packaged as may have a version that is not the one present in the current Maven
        // local repository were Tooling is using to resolve dependencies.
        bundleDescriptorBuilder.setBaseVersion(bundleDescriptor.getBaseVersion());
      }

      bundleDescriptorBuilder
          .setVersion(bundleDescriptor.getVersion())
          .setType(bundleDescriptor.getType())
          .setClassifier(bundleDescriptor.getClassifier().orElse(null));

      return new BundleDependency.Builder()
          .setBundleUri(artifactDescriptor.getClassLoaderModel().getUrls()[0].toURI())
          .sedBundleDescriptor(bundleDescriptorBuilder.build())
          .build();
    } catch (URISyntaxException e) {
      throw new RuntimeException(e);
    }
  }

}
